/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.agent.rest.client.exceptions;

import static java.util.Optional.ofNullable;
import static javax.ws.rs.core.MediaType.APPLICATION_JSON_TYPE;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static org.mule.tooling.agent.rest.client.exceptions.model.ErrorCode.get;
import org.mule.tooling.agent.rest.client.exceptions.model.ErrorCode;
import org.mule.tooling.agent.rest.client.exceptions.model.ErrorEntity;

import java.util.Optional;

import javax.ws.rs.core.Response;

/**
 * Exception for Tooling REST API, it includes as part of the message the {@link Response} body if there is one.
 *
 * @since 4.0
 */
public class ToolingAgentHandlerException extends RuntimeException {

  private ErrorEntity errorEntity;

  private ToolingAgentHandlerException(int statusCode, String reasonPhrase, String causeMessage) {
    super(createMessage(statusCode, reasonPhrase, causeMessage));
  }

  private ToolingAgentHandlerException(int statusCode, String reasonPhrase, ErrorEntity errorEntity) {
    super(createMessage(statusCode, reasonPhrase, errorEntity.toString()));
    this.errorEntity = errorEntity;
  }

  private static String createMessage(int statusCode, String reasonPhrase, String causeMessage) {
    return "Got status code: " + statusCode + " when trying to resolve a Mule Runtime operation. Reason: '" + reasonPhrase
        + (isNotBlank(causeMessage) ? ". " + causeMessage : "") + "'";
  }

  public Optional<ErrorEntity> getErrorEntity() {
    return ofNullable(errorEntity);
  }

  /**
   * Creates a instance of the exception with the given response.
   *
   * @param response {@link Response} from server.
   */
  public static ToolingAgentHandlerException newToolingAgentHandlerException(Response response) {
    int statusCode = response.getStatusInfo().getStatusCode();
    String reasonPhrase = response.getStatusInfo().getReasonPhrase();

    if (response.hasEntity() && response.getMediaType().equals(APPLICATION_JSON_TYPE)) {
      ErrorEntity errorEntity = response.readEntity(ErrorEntity.class);
      return new ToolingAgentHandlerException(statusCode, reasonPhrase, errorEntity);
    } else {
      return new ToolingAgentHandlerException(statusCode, reasonPhrase, response.readEntity(String.class));
    }
  }

}
