/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.agent.rest.client.tooling.sessions;

import static java.lang.String.format;
import static javax.ws.rs.core.MediaType.APPLICATION_JSON_TYPE;
import static javax.ws.rs.core.Response.Status.Family.SUCCESSFUL;
import static org.mule.tooling.agent.rest.client.UriUtils.resolveUri;
import static org.mule.tooling.agent.rest.client.exceptions.ToolingAgentHandlerException.newToolingAgentHandlerException;
import static org.slf4j.LoggerFactory.getLogger;

import org.mule.tooling.api.serialization.Serializer;

import java.net.ConnectException;

import javax.ws.rs.client.Client;
import javax.ws.rs.client.InvocationCallback;
import javax.ws.rs.client.WebTarget;
import javax.ws.rs.core.Response;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.slf4j.Logger;
import org.slf4j.MDC;

public class SessionId {

  private final Logger logger = getLogger(this.getClass());

  static final String REQUEST_ID_HEADER_NAME = "X-B3-TraceId";
  static final String MDC_TRACE_ID_KEY = "traceId";

  private final String id;
  private final String baseUrl;
  private final Client client;

  public Values values;
  public FieldValues fieldValues;
  public Metadata metadata;
  public Connection connection;
  public SampleData sampleData;

  public SessionId(String baseUrl, Client client, Serializer serializer, String id) {
    this.baseUrl = resolveUri(baseUrl, id);
    this.id = id;
    this.values = new Values(this.baseUrl, client, serializer);
    this.fieldValues = new FieldValues(this.baseUrl, client, serializer);
    this.metadata = new Metadata(this.baseUrl, client, serializer);
    this.connection = new Connection(this.baseUrl, client);
    this.sampleData = new SampleData(this.baseUrl, client, serializer);
    this.client = client;
  }

  /**
   * DELETE operation, disposes this session.
   */
  public void delete() {
    WebTarget target = this.client.target(baseUrl);
    final javax.ws.rs.client.Invocation.Builder invocationBuilder = target.request(APPLICATION_JSON_TYPE);
    invocationBuilder.header(REQUEST_ID_HEADER_NAME, MDC.get(MDC_TRACE_ID_KEY));

    invocationBuilder.async().delete(new InvocationCallback<Response>() {

      @Override
      public void completed(Response response) {
        if (response.getStatusInfo().getFamily() != SUCCESSFUL) {
          logger.error("Could not dispose declaration session", newToolingAgentHandlerException(response));
        }
      }

      @Override
      public void failed(Throwable throwable) {
        Throwable rootCause = ExceptionUtils.getRootCause(throwable);
        if (rootCause != null && rootCause instanceof ConnectException) {
          logger.warn(format("Could not dispose declaration session: %s Mule Runtime may has been stopped. %s", id,
                             throwable.getCause().getMessage()));
        } else {
          logger.error("Error while disposing declaration session", throwable);
        }
      }
    });
  }


}
