/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.agent.rest.client;

import static org.hamcrest.core.IsEqual.equalTo;
import static org.junit.Assert.assertThat;
import static org.junit.rules.ExpectedException.none;
import static org.mule.tooling.agent.rest.client.UriUtils.resolveUri;

import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;

public class UriUtilsTestCase {

  private static final String BASE_URI = "http://localhost:8080/app";

  @Rule
  public ExpectedException expectedException = none();

  @Test
  public void encodedUri() {
    assertThat(resolveUri(BASE_URI, "..%252F..%252F..%252F..%252F..%252F.bash_profile%253F"),
               // Would be encoded twice
               equalTo("http://localhost:8080/app/..%25252F..%25252F..%25252F..%25252F..%25252F.bash_profile%25253F"));
  }

  @Test
  public void twiceEncodedUri() {
    assertThat(resolveUri(BASE_URI,
                          "%255c%255c..%255c/..%255c/..%255c/..%255c/..%255c/..%255c/..%255c/..%255c/..%255c/windows/win.ini"),
               // Would be encoded twice
               equalTo("http://localhost:8080/app/%25255c%25255c..%25255c%2F..%25255c%2F..%25255c%2F..%25255c%2F..%25255c%2F..%25255c%2F..%25255c%2F..%25255c%2F..%25255c%2Fwindows%2Fwin.ini"));
  }

  @Test
  public void invalidUriPathTraversal() {
    expectedException.expect(IllegalArgumentException.class);
    expectedException.expectMessage("Invalid URI");
    resolveUri(BASE_URI, "../../pom.xml?");
  }

  @Test
  public void invalidUriPathTraversalUsingQueryParam() {
    assertThat(resolveUri(BASE_URI, "flowName?filename=../../../etc/passwd"),
               equalTo("http://localhost:8080/app/flowName%3Ffilename%3D..%2F..%2F..%2Fetc%2Fpasswd"));
  }

  @Test
  public void invalidUriPathTraversalUsingColon() {
    expectedException.expect(IllegalArgumentException.class);
    expectedException.expectMessage("Invalid URI");
    resolveUri(BASE_URI, "../../pom:xml");
  }

  @Test
  public void invalidUriPathTraversalUsingDot() {
    expectedException.expect(IllegalArgumentException.class);
    expectedException.expectMessage("Invalid URI");
    resolveUri(BASE_URI, "../../pom.xml");
  }

  @Test
  public void validUsingColon() {
    assertThat(resolveUri(BASE_URI, "get:test:test-api-config"),
               equalTo("http://localhost:8080/app/get%3Atest%3Atest-api-config"));
  }

  @Test
  public void validUriWithBackslah() {
    assertThat(resolveUri(BASE_URI, "get:\\test:test-api-config"),
               equalTo("http://localhost:8080/app/get%3A%5Ctest%3Atest-api-config"));
  }

}
