/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.agent.rest.client;

import static java.lang.String.format;
import static java.lang.System.lineSeparator;
import static java.util.Objects.requireNonNull;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static javax.ws.rs.client.ClientBuilder.newBuilder;
import static org.apache.commons.lang3.ClassUtils.getShortClassName;
import static org.glassfish.jersey.client.ClientProperties.CONNECT_TIMEOUT;
import static org.mule.tooling.agent.rest.client.ConnectionValidationResultFactory.failure;
import static org.mule.tooling.agent.rest.client.ConnectionValidationResultFactory.success;
import static org.mule.tooling.agent.rest.client.VersionUtils.isVersionGreaterOrEqualThan;
import static org.mule.tooling.agent.rest.client.api.ToolingServiceAPIClient.create;
import static org.mule.tooling.agent.rest.client.exceptions.model.ErrorCode.BUNDLE_NOT_FOUND;
import static org.mule.tooling.agent.rest.client.exceptions.model.ErrorCode.CONNECTIVITY_TESTING_OBJECT_NOT_FOUND;
import static org.mule.tooling.agent.rest.client.exceptions.model.ErrorCode.NO_SUCH_APPLICATION;
import static org.mule.tooling.agent.rest.client.exceptions.model.ErrorCode.UNKNOWN_ERROR;
import static org.mule.tooling.agent.rest.client.exceptions.model.ErrorCode.UNSUPPORTED_CONNECTIVITY_TESTING_OBJECT;
import static org.mule.tooling.agent.rest.client.service.ServiceFunction.serviceCallWrapper;

import org.mule.runtime.api.meta.model.ComponentModel;
import org.mule.runtime.api.meta.model.operation.OperationModel;
import org.mule.runtime.api.meta.model.source.SourceModel;
import org.mule.runtime.api.metadata.MetadataKeysContainer;
import org.mule.runtime.api.metadata.descriptor.ComponentMetadataDescriptor;
import org.mule.runtime.api.metadata.descriptor.ComponentMetadataTypesDescriptor;
import org.mule.runtime.api.metadata.resolving.MetadataResult;
import org.mule.runtime.api.util.Reference;
import org.mule.runtime.api.value.ValueResult;
import org.mule.runtime.app.declaration.api.ArtifactDeclaration;
import org.mule.runtime.app.declaration.api.ComponentElementDeclaration;
import org.mule.runtime.app.declaration.api.ConfigurationElementDeclaration;
import org.mule.runtime.app.declaration.api.ConnectionElementDeclaration;
import org.mule.runtime.app.declaration.api.ParameterizedElementDeclaration;
import org.mule.runtime.app.declaration.api.ParameterizedElementDeclarationVisitor;
import org.mule.tooling.agent.RuntimeToolingService;
import org.mule.tooling.agent.rest.client.api.ToolingServiceAPIClient;
import org.mule.tooling.agent.rest.client.exceptions.ToolingAgentHandlerException;
import org.mule.tooling.agent.rest.client.exceptions.model.ErrorCode;
import org.mule.tooling.agent.rest.client.exceptions.model.ErrorEntity;
import org.mule.tooling.agent.rest.client.filter.AuthorizationRequestFilter;
import org.mule.tooling.agent.rest.client.tooling.applications.applicationId.components.componentId.connection.model.ConnectivityTestingResponse;
import org.mule.tooling.agent.rest.client.tooling.applications.applicationName.messageHistory.AgentTrackingNotificationResponse;
import org.mule.tooling.agent.rest.client.tooling.applications.model.ApplicationsPUTBody;
import org.mule.tooling.agent.rest.client.tooling.domains.model.DomainsPUTBody;
import org.mule.tooling.api.request.session.DeclarationSessionCreationRequest;
import org.mule.tooling.api.request.values.FieldValuesRequest;
import org.mule.tooling.api.request.values.ValuesRequest;
import org.mule.tooling.api.sampledata.SampleDataMessageModelResult;
import org.mule.tooling.client.api.configuration.agent.proxy.ProxyConfig;
import org.mule.tooling.client.api.configuration.ssl.SslConfiguration;
import org.mule.tooling.client.api.connectivity.BundleNotFoundException;
import org.mule.tooling.client.api.connectivity.ConnectionValidationResult;
import org.mule.tooling.client.api.connectivity.ConnectivityTestingObjectNotFoundException;
import org.mule.tooling.client.api.connectivity.UnsupportedConnectivityTestingObjectException;
import org.mule.tooling.client.api.exception.DeploymentException;
import org.mule.tooling.client.api.exception.NoSuchApplicationException;
import org.mule.tooling.client.api.exception.ServiceUnavailableException;
import org.mule.tooling.client.api.exception.TimeoutException;
import org.mule.tooling.client.api.exception.ToolingException;

import com.mulesoft.agent.domain.tooling.BundleDescriptor;
import com.mulesoft.agent.domain.tooling.dataweave.model.PreviewRequest;
import com.mulesoft.agent.domain.tooling.dataweave.model.PreviewResponse;
import com.mulesoft.agent.external.handlers.tooling.TestConnectionRequest;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.ConnectException;
import java.net.SocketTimeoutException;
import java.net.URL;
import java.security.KeyStore;
import java.security.SecureRandom;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.KeyManagerFactory;
import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManagerFactory;
import javax.ws.rs.ProcessingException;
import javax.ws.rs.client.Client;
import javax.ws.rs.client.ClientBuilder;
import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * REST client for Mule Agent.
 *
 * @since 4.0
 */
public class RestAgentToolingService implements RuntimeToolingService {

  private final Logger logger = LoggerFactory.getLogger(this.getClass());

  private static final String MULE_4_1_5 = "4.1.5-SNAPSHOT";
  private static final String MULE_4_2_0 = "4.2.0-SNAPSHOT";


  private ToolingServiceAPIClient client;
  private URL url;
  private int defaultReadTimeout;
  private static boolean VERBOSE_ERROR_ENABLED = true;
  private Optional<SSLContext> sslContext;

  private String muleVersion;
  private String authorizationToken;

  /**
   * {@inheritDoc}
   */
  @Override
  public void setToolingApiUrl(URL url, long defaultConnectTimeout, long defaultReadTimeout,
                               Optional<SslConfiguration> sslConfiguration, Optional<ProxyConfig> proxyConfig,
                               String authorizationToken) {
    requireNonNull(url, "url cannot be null");

    if (logger.isDebugEnabled()) {
      logger.debug("Configuring using tooling API URL:{}, defaultConnectTimeout:{}ms, defaultReadTimeout:{}ms", url,
                   defaultConnectTimeout,
                   defaultReadTimeout);
    }
    this.sslContext = sslConfiguration.map(configuration -> {
      try {
        InputStream keyStoreFile = new FileInputStream(configuration.getKeyStoreConfig().getKeyStoreFile());
        InputStream trustStoreFile = new FileInputStream(configuration.getTrustStoreConfig().getTrustStoreFile());

        // TODO (gfernandes) this code should be also extensible by clients...
        KeyStore trustStore = KeyStore.getInstance("JKS");
        trustStore.load(trustStoreFile, configuration.getKeyStoreConfig().getKeyStorePassword().toCharArray());
        KeyStore keyStore = KeyStore.getInstance("JKS");
        keyStore.load(keyStoreFile, configuration.getTrustStoreConfig().getTrustStorePassword().toCharArray());

        SSLContext ctx = SSLContext.getInstance("TLSv1.2");
        KeyManagerFactory sunX509 = KeyManagerFactory.getInstance("SunX509");
        sunX509.init(keyStore, configuration.getKeyStoreConfig().getKeyStorePassword().toCharArray());
        TrustManagerFactory trustManagerFactory = TrustManagerFactory.getInstance("SunX509", "SunJSSE");
        trustManagerFactory.init(trustStore);
        ctx.init(sunX509.getKeyManagers(), trustManagerFactory.getTrustManagers(), new SecureRandom());

        HttpsURLConnection.setDefaultHostnameVerifier((s, sslSession) -> true);
        return of(ctx);
      } catch (Exception e) {
        throw new ToolingException("Error while configuring mutual SSL to connect to the Mule Agent REST API.", e);
      }
    }).orElse(empty());
    this.defaultReadTimeout = Long.valueOf(defaultReadTimeout).intValue();
    this.client = create(url.toString(), Long.valueOf(defaultConnectTimeout).intValue(), this.defaultReadTimeout, sslContext,
                         proxyConfig, authorizationToken);
    this.authorizationToken = authorizationToken;
    this.url = url;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isOperational() {
    ClientBuilder builder = newBuilder().property(CONNECT_TIMEOUT, 1000);
    sslContext.ifPresent(sslContext -> builder.sslContext(sslContext));
    Client client = builder.register(new AuthorizationRequestFilter(authorizationToken)).build();
    try {
      client.target(this.url.toString()).request().buildGet().invoke();
      return true;
    } catch (ProcessingException e) {
      if (e.getCause() instanceof ConnectException) {
        return false;
      }
      // In case of another error just propagate the exception
      throw new ToolingException("Error while checking if Mule Agent REST service is available", e);
    }
  }

  @Override
  public void setMuleVersion(String muleVersion) {
    this.muleVersion = muleVersion;
  }

  public List<String> listApplicationsId() {
    if (logger.isDebugEnabled()) {
      logger.debug("GET:tooling/applications/");
    }

    try {
      return serviceCallWrapper(() -> client.tooling.applications
          .get(VERBOSE_ERROR_ENABLED, defaultReadTimeout).stream()
          .map(applicationsGETResponse -> applicationsGETResponse.getApplicationId())
          .collect(Collectors.toList()), createDescription("list(applications)"));
    } catch (ToolingAgentHandlerException e) {
      throw handleToolingAgentHandlerException(e);
    } catch (Exception e) {
      throw serviceExceptionOrToolingException("Unknown error while getting retrieving applications", e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String deployApplication(String id, File appLocation, String domainName, Map<String, String> deploymentProperties) {
    if (logger.isDebugEnabled()) {
      logger.debug("PUT:tooling/applications/{} appLocation:[{}], domain:{}", id, appLocation,
                   domainName != null ? domainName : "");
    }

    try {
      if (supportsDeploymentProperties()) {
        return serviceCallWrapper(() -> client.tooling.applications
            .put(new ApplicationsPUTBody().withId(id).withAppLocation(appLocation.getAbsolutePath())
                .withDomainName(domainName).withDeploymentProperties(deploymentProperties), VERBOSE_ERROR_ENABLED,
                 defaultReadTimeout)
            .getApplicationId(), createDescription(format("deploy(application:%s)", id)));
      }
      return serviceCallWrapper(() -> client.tooling.applications
          .put(new ApplicationsPUTBody(id, appLocation.getAbsolutePath(), domainName), VERBOSE_ERROR_ENABLED, defaultReadTimeout)
          .getApplicationId(), createDescription(format("deploy(application:%s)", id)));
    } catch (Exception e) {
      throw serviceExceptionOr(e, () -> newDeploymentException(format("Couldn't deploy the application: '%s'", id), e));
    }
  }

  private String createDescription(String method) {
    return RuntimeToolingService.class.getSimpleName() + "#" + method;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String deployDomain(String id, File domainLocation, Map<String, String> deploymentProperties) {
    if (logger.isDebugEnabled()) {
      logger.debug("PUT:tooling/domains/{} domainLocation:[{}]", id, domainLocation);
    }

    try {
      if (supportsDeploymentProperties()) {
        return serviceCallWrapper(() -> client.tooling.domains
            .put(new DomainsPUTBody().withId(id).withDomainLocation(domainLocation.getAbsolutePath())
                .withDeploymentProperties(deploymentProperties),
                 VERBOSE_ERROR_ENABLED, defaultReadTimeout)
            .getDomainId(), createDescription(format("deploy(domain:%s)", id)));
      }
      return serviceCallWrapper(() -> client.tooling.domains
          .put(new DomainsPUTBody().withId(id).withDomainLocation(domainLocation.getAbsolutePath()), VERBOSE_ERROR_ENABLED,
               defaultReadTimeout)
          .getDomainId(), createDescription(format("deploy(domain:%s)", id)));
    } catch (Exception e) {
      throw serviceExceptionOr(e, () -> newDeploymentException(format("Couldn't deploy the domain: '%s'", id), e));
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String deployDomain(String id, InputStream inputStream, Map<String, String> deploymentProperties)
      throws DeploymentException, ServiceUnavailableException {
    if (logger.isDebugEnabled()) {
      logger.debug("PUT:tooling/domains/{} inputStream, id");
    }

    try {
      if (supportsDeploymentProperties()) {
        return serviceCallWrapper(() -> client.tooling.domains
            .put(new DomainsPUTBody().withId(id).withContent(toByteArray(inputStream))
                .withDeploymentProperties(deploymentProperties),
                 VERBOSE_ERROR_ENABLED, defaultReadTimeout)
            .getDomainId(), createDescription("deploy(domain)"));
      }
      return serviceCallWrapper(() -> client.tooling.domains
          .put(inputStream, id, VERBOSE_ERROR_ENABLED, defaultReadTimeout)
          .getDomainId(), createDescription("deploy(domain)"));
    } catch (Exception e) {
      throw serviceExceptionOr(e, () -> newDeploymentException(format("Couldn't deploy the domain: '%s'", id), e));
    }
  }

  private byte[] toByteArray(InputStream inputStream) {
    try {
      return IOUtils.toByteArray(inputStream);
    } catch (IOException e) {
      throw new ToolingException(e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String deployApplication(String id, InputStream inputStream, Map<String, String> deploymentProperties)
      throws DeploymentException, ServiceUnavailableException {
    if (logger.isDebugEnabled()) {
      logger.debug("PUT:tooling/applications/{} inputStream", id);
    }

    try {
      if (supportsDeploymentProperties()) {
        return serviceCallWrapper(() -> client.tooling.applications
            .put(new ApplicationsPUTBody().withId(id).withContent(toByteArray(inputStream))
                .withDeploymentProperties(deploymentProperties), VERBOSE_ERROR_ENABLED, defaultReadTimeout)
            .getApplicationId(), createDescription("deploy(application)"));
      }
      return serviceCallWrapper(() -> client.tooling.applications
          .put(inputStream, id, VERBOSE_ERROR_ENABLED, defaultReadTimeout)
          .getApplicationId(), createDescription("deploy(application)"));

    } catch (Exception e) {
      throw serviceExceptionOr(e, () -> newDeploymentException(format("Couldn't deploy the application: '%s'", id), e));
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String deployApplication(String id, InputStream inputStream, String domainName, Map<String, String> deploymentProperties)
      throws DeploymentException, ServiceUnavailableException {
    if (logger.isDebugEnabled()) {
      logger.debug("PUT:tooling/applications/{} inputStream", id);
    }

    try {
      if (supportsDeploymentProperties()) {
        return serviceCallWrapper(() -> client.tooling.applications
            .put(new ApplicationsPUTBody().withId(id).withContent(toByteArray(inputStream))
                .withDomainName(domainName).withDeploymentProperties(deploymentProperties), VERBOSE_ERROR_ENABLED,
                 defaultReadTimeout)
            .getApplicationId(), createDescription("deploy(application)"));
      }
      return serviceCallWrapper(() -> client.tooling.applications
          .put(inputStream, id, domainName, VERBOSE_ERROR_ENABLED, defaultReadTimeout)
          .getApplicationId(), createDescription("deploy(application)"));
    } catch (Exception e) {
      throw serviceExceptionOr(e, () -> newDeploymentException(format("Couldn't deploy the application: '%s'", id), e));
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void disposeApplication(String applicationId) throws ServiceUnavailableException {
    if (logger.isDebugEnabled()) {
      logger.debug("DELETE:tooling/applications/{}", applicationId);
    }
    try {
      serviceCallWrapper(() -> client.tooling.applications.applicationId(applicationId).delete(VERBOSE_ERROR_ENABLED,
                                                                                               defaultReadTimeout),
                         createDescription("dispose(application)"));
    } catch (Exception e) {
      throw serviceExceptionOrToolingException(format("Unknown error while disposing application, for applicationId: %s",
                                                      applicationId),
                                               e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void disposeDomain(String domainId) throws ServiceUnavailableException {
    if (logger.isDebugEnabled()) {
      logger.debug("DELETE:tooling/domains/{}", domainId);
    }
    try {
      serviceCallWrapper(() -> client.tooling.domains.domainId(domainId).delete(VERBOSE_ERROR_ENABLED, defaultReadTimeout),
                         createDescription("dispose(domain)"));
    } catch (Exception e) {
      throw serviceExceptionOrToolingException(format("Unknown error while disposing domain, for domainId: %s", domainId), e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ConnectionValidationResult testConnectionApplication(String applicationId, String componentId, long readTimeout)
      throws NoSuchApplicationException,
      UnsupportedConnectivityTestingObjectException, ConnectivityTestingObjectNotFoundException, ServiceUnavailableException {
    if (logger.isDebugEnabled()) {
      logger.debug("GET:tooling/applications/{}/components/{}/connection", applicationId, componentId);
    }
    try {
      ConnectivityTestingResponse response =
          serviceCallWrapper(() -> client.tooling.applications.applicationId(applicationId).components
              .componentId(componentId).connection
                  .get(VERBOSE_ERROR_ENABLED, Long.valueOf(readTimeout).intValue()),
                             createDescription("testConnection(application)"));
      return response.getValidationStatus() ? success()
          : failure(response.getMessage(), response.getErrorType(), response.getException());
    } catch (Exception e) {
      throw serviceExceptionOrToolingException("Unknown error while doing connectivity testing on application", e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ConnectionValidationResult testConnectionDomain(String domainId, String componentId, long readTimeout)
      throws NoSuchApplicationException,
      UnsupportedConnectivityTestingObjectException, ConnectivityTestingObjectNotFoundException, ServiceUnavailableException {
    if (logger.isDebugEnabled()) {
      logger.debug("GET:tooling/domains/{}/components/{}/connection", domainId, componentId);
    }
    try {
      ConnectivityTestingResponse response =
          serviceCallWrapper(() -> client.tooling.domains.domainId(domainId).components.componentId(componentId).connection
              .get(VERBOSE_ERROR_ENABLED, Long.valueOf(readTimeout).intValue()), createDescription("testConnection(domain)"));
      return response.getValidationStatus() ? success()
          : failure(response.getMessage(), response.getErrorType(), response.getException());
    } catch (Exception e) {
      throw serviceExceptionOrToolingException("Unknown error while doing connectivity testing on application", e);
    }
  }

  @Override
  public ConnectionValidationResult testConnection(List<BundleDescriptor> dependencies, ArtifactDeclaration artifactDeclaration,
                                                   String componentId, long readTimeout)
      throws UnsupportedConnectivityTestingObjectException, ConnectivityTestingObjectNotFoundException,
      ServiceUnavailableException {
    if (logger.isDebugEnabled()) {
      logger.debug("PUT:tooling/components/{}/connection", componentId);
    }
    try {
      ConnectivityTestingResponse response =
          serviceCallWrapper(() -> client.tooling.components.componentId(componentId).connection
              .put(createRequest(dependencies, artifactDeclaration), VERBOSE_ERROR_ENABLED, Long.valueOf(readTimeout).intValue()),
                             createDescription("testConnection(temporaryArtifact)"));
      return response.getValidationStatus() ? success()
          : failure(response.getMessage(), response.getErrorType(), response.getException());
    } catch (Exception e) {
      throw serviceExceptionOrToolingException("Unknown error while doing connectivity testing on application", e);
    }
  }

  @Override
  public PreviewResponse runDataWeaveApplication(String applicationId, PreviewRequest request) {
    if (logger.isDebugEnabled()) {
      logger.debug("POST:tooling/applications/{}/dataweave/execute", applicationId);
    }
    try {
      PreviewResponse response =
          serviceCallWrapper(() -> client.tooling.applications.applicationId(applicationId).dataweave.execute
              .post(request, VERBOSE_ERROR_ENABLED,
                    defaultReadTimeout), createDescription("runDataWeave(application)"));
      return response;
    } catch (Exception e) {
      throw serviceExceptionOrToolingException("Unknown error while executing dataweave script", e);
    }
  }

  @Override
  public PreviewResponse runDataWeaveDomain(String domainId, PreviewRequest request) {
    if (logger.isDebugEnabled()) {
      logger.debug("POST:tooling/domains/{}/dataweave/execute", domainId);
    }
    try {
      PreviewResponse response =
          serviceCallWrapper(() -> client.tooling.domains.domainId(domainId).dataweave.execute.post(request,
                                                                                                    VERBOSE_ERROR_ENABLED,
                                                                                                    defaultReadTimeout),
                             createDescription("runDataWeave(domain)"));
      return response;
    } catch (Exception e) {
      throw serviceExceptionOrToolingException("Unknown error while executing dataweave script", e);
    }
  }

  private TestConnectionRequest createRequest(List<BundleDescriptor> dependencies, ArtifactDeclaration artifactDeclaration) {
    TestConnectionRequest request = new TestConnectionRequest();
    request.setArtifactDeclaration(artifactDeclaration);
    request.setDependencies(dependencies);
    return request;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public MetadataResult<MetadataKeysContainer> getMetadataKeysApplication(String applicationId, String componentLocation,
                                                                          long readTimeout)
      throws ServiceUnavailableException {
    if (logger.isDebugEnabled()) {
      logger.debug("GET:tooling/applications/{}/components/{}/keys", applicationId, componentLocation);
    }
    try {
      return serviceCallWrapper(() -> client.tooling.applications.applicationId(applicationId).components
          .componentId(componentLocation).keys
              .get(VERBOSE_ERROR_ENABLED, Long.valueOf(readTimeout).intValue()),
                                createDescription("getMetadataKeys(application)"));
    } catch (Exception e) {
      throw serviceExceptionOrToolingException("Unknown error while getting Metadata keys", e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public MetadataResult<MetadataKeysContainer> getMetadataKeysDomain(String domainId, String componentLocation, long readTimeout)
      throws ServiceUnavailableException {
    if (logger.isDebugEnabled()) {
      logger.debug("GET:tooling/domains/{}/components/{}/keys", domainId, componentLocation);
    }
    try {
      return serviceCallWrapper(() -> client.tooling.domains.domainId(domainId).components.componentId(componentLocation).keys
          .get(VERBOSE_ERROR_ENABLED, Long.valueOf(readTimeout).intValue()),
                                createDescription("getMetadataKeys(domain)"));
    } catch (Exception e) {
      throw serviceExceptionOrToolingException("Unknown error while getting Metadata keys", e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public MetadataResult<ComponentMetadataDescriptor<OperationModel>> getOperationMetadata(String applicationId,
                                                                                          String componentLocation,
                                                                                          long readTimeout)
      throws ServiceUnavailableException {
    return getMetadata(applicationId, componentLocation, readTimeout);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public MetadataResult<ComponentMetadataDescriptor<SourceModel>> getSourceMetadata(String applicationId,
                                                                                    String componentLocation,
                                                                                    long readTimeout)
      throws ServiceUnavailableException {
    return getMetadata(applicationId, componentLocation, readTimeout);
  }

  private <T extends ComponentModel> MetadataResult<ComponentMetadataDescriptor<T>> getMetadata(String applicationId,
                                                                                                String componentLocation,
                                                                                                long readTimeout) {
    if (logger.isDebugEnabled()) {
      logger.debug("GET:tooling/applications/{}/components/{}/metadata", applicationId, componentLocation);
    }
    try {
      return serviceCallWrapper(() -> client.tooling.applications
          .applicationId(applicationId).components
              .componentId(componentLocation).metadata.get(VERBOSE_ERROR_ENABLED,
                                                           Long.valueOf(readTimeout).intValue(),
                                                           supportsReducedJsonSerialization()),
                                createDescription("getMetadata"));
    } catch (Exception e) {
      throw serviceExceptionOrToolingException("Unknown error while getting Metadata", e);
    }
  }

  @Override
  public void disposeApplicationMetadataCache(String applicationId, String hashKey) {
    if (logger.isDebugEnabled()) {
      logger.debug("DELETE:tooling/applications/{}/cache/{}", applicationId, hashKey);
    }
    doDisposeMetadataCache(() -> {
      client.tooling.applications.applicationId(applicationId).cache.hashKey(hashKey).delete(VERBOSE_ERROR_ENABLED);
      return null;
    }, createDescription("disposeApplicationMetadataCache"), "Error while disposing Metadata");
  }

  private <OutputType> OutputType doDisposeMetadataCache(Supplier<OutputType> function, String serviceMethod,
                                                         String errorMessage) {
    try {
      return serviceCallWrapper(function, createDescription(serviceMethod));
    } catch (Exception e) {
      throw serviceExceptionOrToolingException("Unknown error while disposing Metadata cache", e);
    }
  }

  @Override
  public void disposeDomainMetadataCache(String domainId, String hashKey) {
    if (logger.isDebugEnabled()) {
      logger.debug("DELETE:tooling/domains/{}/cache/{}", domainId, hashKey);
    }
    doDisposeMetadataCache(() -> {
      client.tooling.domains.domainId(domainId).cache.hashKey(hashKey).delete(VERBOSE_ERROR_ENABLED);
      return null;
    }, createDescription("disposeDomainMetadataCache"), "Error while disposing Metadata");
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void enableMessageHistory(String applicationName) throws NoSuchApplicationException {
    if (logger.isDebugEnabled()) {
      logger.debug("PUT:tooling/applications/{}/messagehistory/", applicationName);
    }
    try {
      client.tooling.applications.applicationName(applicationName).messageHistory
          .put(VERBOSE_ERROR_ENABLED, defaultReadTimeout);
    } catch (Exception e) {
      throw serviceExceptionOrToolingException("Unknown error while enabling application for message history", e);
    }
  }

  @Override
  public List<AgentTrackingNotificationResponse> consumeMessageHistoryNotifications(String applicationName, int chunkSize) {
    if (logger.isDebugEnabled()) {
      logger.debug("GET:tooling/applications/{}/messagehistory/", applicationName);
    }
    try {
      return client.tooling.applications.applicationName(applicationName).messageHistory.get(chunkSize, VERBOSE_ERROR_ENABLED,
                                                                                             defaultReadTimeout);
    } catch (Exception e) {
      throw serviceExceptionOrToolingException("Unknown error while getting application notifications for message history", e);
    }
  }

  @Override
  public void disableMessageHistory(String applicationName) {
    if (logger.isDebugEnabled()) {
      logger.debug("DELETE:tooling/applications/{}/messagehistory/", applicationName);
    }
    try {
      client.tooling.applications.applicationName(applicationName).messageHistory.delete(VERBOSE_ERROR_ENABLED,
                                                                                         defaultReadTimeout);
    } catch (Exception e) {
      throw serviceExceptionOrToolingException("Unknown error while disabling application for message history", e);
    }
  }

  @Override
  public ValueResult getValuesApplication(String applicationId, String location, String providerName) {
    if (logger.isDebugEnabled()) {
      logger.debug("GET:tooling/applications/{}/components/{}/valueProviders/{}", applicationId, location, providerName);
    }
    try {
      return serviceCallWrapper(() -> client.tooling.applications.applicationId(applicationId).components
          .componentId(location).valueProviders
              .get(providerName, defaultReadTimeout), createDescription("getValues(application"));
    } catch (Exception e) {
      throw serviceExceptionOrToolingException(format("Unknown error while trying to resolve Values for the ValueProvider [%s] located in [%s] of the application with ID [%s]",
                                                      providerName, location, applicationId),
                                               e);
    }
  }

  @Override
  public ValueResult getValuesDomain(String domainId, String location, String providerName) {
    if (logger.isDebugEnabled()) {
      logger.debug("GET:tooling/domains/{}/components/{}/valueProviders/{}", domainId, location, providerName);
    }
    try {
      return serviceCallWrapper(() -> client.tooling.domains.domainId(domainId).components.componentId(location).valueProviders
          .get(providerName, defaultReadTimeout), createDescription("getValues(domain)"));
    } catch (Exception e) {
      throw serviceExceptionOrToolingException(format("Unknown error while trying to resolve Values for the ValueProvider [%s] located in [%s] of the domain with ID [%s]",
                                                      providerName, location, domainId),
                                               e);
    }
  }

  @Override
  public SampleDataMessageModelResult getSampleDataApplication(String applicationId, String componentLocation, long readTimeout)
      throws ServiceUnavailableException {
    if (logger.isDebugEnabled()) {
      logger.debug("GET:tooling/applications/{}/components/{}/sampledata", applicationId, componentLocation);
    }
    try {
      return serviceCallWrapper(() -> client.tooling.applications
          .applicationId(applicationId).components
              .componentId(componentLocation).sampleData
                  .get(VERBOSE_ERROR_ENABLED, Long.valueOf(readTimeout).intValue()),
                                createDescription("getSampleData"));
    } catch (Exception e) {
      throw serviceExceptionOrToolingException("Unknown error while getting Metadata", e);
    }
  }

  @Override
  public String createDeclarationSession(DeclarationSessionCreationRequest declarationSessionCreationRequest) {
    try {
      return serviceCallWrapper(() -> client.tooling.sessions.post(declarationSessionCreationRequest, VERBOSE_ERROR_ENABLED),
                                createDescription("newDeclarationSession()"));
    } catch (Exception e) {
      throw serviceExceptionOrToolingException("Error while creating a new DeclarationSession", e);
    }
  }

  @Override
  public void disposeDeclarationSession(String sessionId) {
    try {
      serviceCallWrapper(() -> client.tooling.sessions.sessionId(sessionId).delete(),
                         createDescription("disposeDeclarationSession()"));
    } catch (Exception e) {
      throw serviceExceptionOrToolingException(format("Error while disposing a DeclarationSession with id: '%s'",
                                                      sessionId),
                                               e);
    }
  }

  @Override
  public ValueResult getValues(String sessionId,
                               ParameterizedElementDeclaration parameterizedElementDeclaration,
                               String providerName) {
    try {
      Reference<ValuesRequest> valuesRequestReference = new Reference<>();
      parameterizedElementDeclaration.accept(new ParameterizedElementDeclarationVisitor() {

        @Override
        public void visitConfigurationElementDeclaration(ConfigurationElementDeclaration configurationElementDeclaration) {
          valuesRequestReference.set(new ValuesRequest(configurationElementDeclaration, providerName));
        }

        @Override
        public void visitConnectionElementDeclaration(ConnectionElementDeclaration connectionElementDeclaration) {
          valuesRequestReference.set(new ValuesRequest(connectionElementDeclaration, providerName));
        }

        @Override
        public void visitComponentElementDeclaration(ComponentElementDeclaration componentElementDeclaration) {
          valuesRequestReference.set(new ValuesRequest(componentElementDeclaration, providerName));
        }
      });

      return serviceCallWrapper(() -> client.tooling.sessions.sessionId(sessionId).values
          .put(valuesRequestReference.get(), VERBOSE_ERROR_ENABLED),
                                createDescription("getValues()"));
    } catch (Exception e) {
      throw serviceExceptionOrToolingException(format("Error while getting values for providerName: '%s', element: '%s', extension: '%s', session: '%s'",
                                                      providerName, parameterizedElementDeclaration.getName(),
                                                      parameterizedElementDeclaration.getDeclaringExtension(),
                                                      sessionId),
                                               e);
    }
  }

  @Override
  public ValueResult getFieldValues(String sessionId,
                                    ParameterizedElementDeclaration parameterizedElementDeclaration,
                                    String providerName,
                                    String targetSelector) {
    try {
      final Reference<FieldValuesRequest> fieldValuesRequestReference = new Reference<>();
      parameterizedElementDeclaration.accept(new ParameterizedElementDeclarationVisitor() {

        @Override
        public void visitComponentElementDeclaration(ComponentElementDeclaration componentElementDeclaration) {
          fieldValuesRequestReference.set(new FieldValuesRequest(componentElementDeclaration, providerName, targetSelector));
        }
      });

      return serviceCallWrapper(
                                () -> client.tooling.sessions
                                    .sessionId(sessionId).fieldValues
                                        .put(fieldValuesRequestReference.get(), VERBOSE_ERROR_ENABLED),
                                createDescription("getFieldValues()"));
    } catch (Exception e) {
      throw serviceExceptionOrToolingException(format("Error while getting field values for providerName: '%s', targetSelector: '%s', element: '%s', extension: '%s', session: '%s'",
                                                      providerName,
                                                      targetSelector,
                                                      parameterizedElementDeclaration.getName(),
                                                      parameterizedElementDeclaration.getDeclaringExtension(),
                                                      sessionId),
                                               e);
    }
  }

  @Override
  public MetadataResult<ComponentMetadataTypesDescriptor> getComponentMetadata(String sessionId,
                                                                               ComponentElementDeclaration componentElementDeclaration,
                                                                               boolean ignoreCache) {
    try {
      return serviceCallWrapper(() -> client.tooling.sessions.sessionId(sessionId).metadata
          .put(componentElementDeclaration, ignoreCache, VERBOSE_ERROR_ENABLED),
                                createDescription("getComponentMetadata()"));
    } catch (Exception e) {
      throw serviceExceptionOrToolingException(format("Error while resolving metadata for element: '%s', extension: '%s', session: '%s'",
                                                      componentElementDeclaration.getName(),
                                                      componentElementDeclaration.getDeclaringExtension(),
                                                      sessionId),
                                               e);
    }
  }

  @Override
  public MetadataResult<MetadataKeysContainer> getComponentMetadataKeys(String sessionId,
                                                                        ComponentElementDeclaration componentElementDeclaration,
                                                                        boolean ignoreCache) {
    try {
      return serviceCallWrapper(() -> client.tooling.sessions.sessionId(sessionId).metadata.keys
          .put(componentElementDeclaration, ignoreCache, VERBOSE_ERROR_ENABLED),
                                createDescription("getComponentMetadataKeys()"));
    } catch (Exception e) {
      throw serviceExceptionOrToolingException(format("Error while getting metadata keys from element: '%s', extension: '%s', session: '%s'",
                                                      componentElementDeclaration.getDeclaringExtension(),
                                                      componentElementDeclaration.getName(),
                                                      sessionId),
                                               e);
    }
  }

  @Override
  public SampleDataMessageModelResult getSampleData(String sessionId,
                                                    ComponentElementDeclaration componentElementDeclaration) {
    try {
      return serviceCallWrapper(() -> client.tooling.sessions.sessionId(sessionId).sampleData
          .put(componentElementDeclaration, VERBOSE_ERROR_ENABLED),
                                createDescription("getSampleData()"));
    } catch (Exception e) {
      throw serviceExceptionOrToolingException(format("Error while getting metadata keys from element: '%s', extension: '%s', session: '%s'",
                                                      componentElementDeclaration.getDeclaringExtension(),
                                                      componentElementDeclaration.getName(),
                                                      sessionId),
                                               e);
    }
  }


  @Override
  public ConnectionValidationResult testConnection(String sessionId, String configName) {
    try {
      ConnectivityTestingResponse response =
          serviceCallWrapper(() -> client.tooling.sessions.sessionId(sessionId).connection.get(configName, VERBOSE_ERROR_ENABLED),
                             createDescription("testConnection"));
      return response.getValidationStatus() ? success()
          : failure(response.getMessage(), response.getErrorType(), response.getException());
    } catch (Exception e) {
      throw serviceExceptionOrToolingException(format("Error while doing connectivity testing on config: '%s', session: '%s'",
                                                      configName, sessionId),
                                               e);
    }
  }

  private boolean supportsDeploymentProperties() {
    return muleVersion != null && isVersionGreaterOrEqualThan(muleVersion, MULE_4_1_5);
  }

  private boolean supportsReducedJsonSerialization() {
    return muleVersion != null && isVersionGreaterOrEqualThan(muleVersion, MULE_4_2_0);
  }
  // **************************
  // Exception handling!
  // **************************

  private ToolingException serviceExceptionOrToolingException(String message, Exception e) {
    return serviceExceptionOr(e, () -> {
      if (e instanceof ToolingAgentHandlerException) {
        return handleToolingAgentHandlerException((ToolingAgentHandlerException) e);
      }
      return new ToolingException(message, e);
    });
  }

  private ToolingException serviceExceptionOr(Exception e, Supplier<ToolingException> supplier) {
    if (e.getCause() instanceof ConnectException) {
      return new ServiceUnavailableException("Mule Agent REST service is not available", e);
    }
    if (e.getCause() instanceof SocketTimeoutException) {
      return new TimeoutException("Mule Agent REST service timed out", e);
    }

    return supplier.get();
  }

  private DeploymentException newDeploymentException(String message, Exception e) {
    if (e instanceof ToolingAgentHandlerException) {
      ToolingAgentHandlerException toolingAgentHandlerException = (ToolingAgentHandlerException) e;
      if (toolingAgentHandlerException.getErrorEntity().isPresent()) {
        ErrorEntity errorEntity = toolingAgentHandlerException.getErrorEntity().get();
        return new DeploymentException(message + ". Message: " + e.getMessage(),
                                       errorEntity.getErrorMessage(), errorEntity.getErrorType(),
                                       errorDetailToStringArray(errorEntity));
      }
    }
    return new DeploymentException(e.getMessage(), e);
  }

  private ToolingException handleToolingAgentHandlerException(ToolingAgentHandlerException e) {
    Optional<ErrorEntity> errorEntityOptional = e.getErrorEntity();
    if (!errorEntityOptional.isPresent()) {
      return new ToolingException(e.getMessage(), e);
    }

    ErrorEntity errorEntity = errorEntityOptional.get();
    final ErrorCode errorCode = errorEntityOptional.map(entity -> ErrorCode.get(entity.getErrorType())).orElse(UNKNOWN_ERROR);
    if (errorCode == NO_SUCH_APPLICATION) {
      return new NoSuchApplicationException("Application resource not found. Message: " + errorEntity.getErrorMessage(), e);
    } else if (errorCode == BUNDLE_NOT_FOUND) {
      return new BundleNotFoundException("Mule Runtime could not find an artifact. Message: " + errorEntity.getErrorMessage(), e);
    } else if (errorCode == CONNECTIVITY_TESTING_OBJECT_NOT_FOUND) {
      return new ConnectivityTestingObjectNotFoundException("Object not found. Message: " + errorEntity.getErrorMessage(), e);
    } else if (errorCode == UNSUPPORTED_CONNECTIVITY_TESTING_OBJECT) {
      return new UnsupportedConnectivityTestingObjectException("Component doesn't support connectivity testing");
    } else {
      return ToolingException.builder()
          .withMessage(e.getMessage())
          .withRootCauseMessage(errorEntity.getErrorMessage())
          .withRootCauseType(errorEntity.getErrorType())
          .withRootCauseStackTrace(errorDetailToStringArray(errorEntity))
          .build();
    }
  }

  private String[] errorDetailToStringArray(ErrorEntity errorEntity) {
    return errorEntity.getErrorDetail() != null ? errorEntity.getErrorDetail().split(lineSeparator())
        : new String[0];
  }

  @Override
  public String toString() {
    return format("%s{url=%s}", getShortClassName(this.getClass()), url);
  }
}
