/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.event.model;

import static java.util.Objects.requireNonNull;
import static org.mule.tooling.event.model.ModelType.collectionModelType;
import static org.mule.tooling.event.model.ModelType.simpleModelType;

public class DataTypeModel {

  private String type;
  private String mediaType;
  private ModelType modelType;

  // Used by json serializer/deserialzer
  public DataTypeModel() {}

  protected DataTypeModel(String type, String mediaType, ModelType modelType) {
    requireNonNull(type, "type cannot be null");
    requireNonNull(mediaType, "mediaType cannot be null");
    requireNonNull(modelType, "modelType cannot be null");

    this.type = type;
    this.mediaType = mediaType;
    this.modelType = modelType;
  }

  public String getMediaType() {
    return mediaType;
  }

  public String getType() {
    return type;
  }

  public ModelType getModelType() {
    return modelType;
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }
    if (o == null || getClass() != o.getClass()) {
      return false;
    }

    DataTypeModel that = (DataTypeModel) o;

    if (!type.equals(that.type)) {
      return false;
    }
    if (!modelType.equals(that.modelType)) {
      return false;
    }
    return mediaType.equals(that.mediaType);
  }

  @Override
  public int hashCode() {
    int result = type.hashCode();
    result = 31 * result + mediaType.hashCode();
    return result;
  }

  public static Builder builder() {
    return new Builder();
  }

  public static class Builder {

    private String type;
    private String mediaType;
    private ModelType modelType = simpleModelType();

    private Builder() {}

    public Builder withType(String type) {
      this.type = type;
      return this;
    }

    public Builder withMediaType(String mediaType) {
      this.mediaType = mediaType;
      return this;
    }

    public Builder asCollectionWithItemsType(String itemsType) {
      modelType = collectionModelType(itemsType);
      return this;
    }

    public DataTypeModel build() {
      return new DataTypeModel(this.type, this.mediaType, modelType);
    }

  }

}
