/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.event.model;

import static java.util.Objects.requireNonNull;

import java.util.HashMap;
import java.util.Map;

/**
 * Represents the Result of the Event, Message, Attributes and Error related to a Mule Runtime Event.
 *
 * @since 1.0
 */
public class EventModel {

  private MessageModel message;
  private Map<String, TypedValueModel> variables = new HashMap<>();
  private ErrorModel error;
  private boolean successful;

  // Used by json serializer/deserializer
  public EventModel() {}

  private EventModel(MessageModel message, Map<String, TypedValueModel> variables, ErrorModel error, boolean isSuccessful) {
    requireNonNull(message, "message cannot be null");

    this.message = message;
    this.variables = variables;
    this.error = error;
    this.successful = isSuccessful;
  }

  /**
   * @return the {@link MessageModel} associated to the event
   */
  public MessageModel getMessage() {
    return message;
  }

  /**
   * @return the Variables involved in the event.
   */
  public Map<String, TypedValueModel> getVariables() {
    return variables;
  }

  /**
   * @return an {@link ErrorModel} for the errors related to this event.
   * In case there was no error for this event, this method will return null.
   * Even if the event is successful (i.e. {@link EventModel#isSuccessful()} is true)
   * there could be an error related to this event.
   */
  public ErrorModel getError() {
    return error;
  }

  /**
   * @return whether the event is successful or not.
   */
  public boolean isSuccessful() {
    return successful;
  }

  public void setSuccessful(boolean successful) {
    this.successful = successful;
  }

  /**
   * @return a new {@link EventModel.Builder} to create a {@link EventModel}.
   */
  public static Builder builder() {
    return new Builder();
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }
    if (o == null || getClass() != o.getClass()) {
      return false;
    }

    EventModel that = (EventModel) o;

    if (!message.equals(that.message)) {
      return false;
    }
    if (variables != null ? !variables.equals(that.variables) : that.variables != null) {
      return false;
    }
    return error != null ? error.equals(that.error) : that.error == null;
  }

  @Override
  public int hashCode() {
    int result = message.hashCode();
    result = 31 * result + (variables != null ? variables.hashCode() : 0);
    result = 31 * result + (error != null ? error.hashCode() : 0);
    return result;
  }

  /**
   * Builder to create an {@link EventModel}
   */
  public static class Builder {

    private MessageModel message;
    private Map<String, TypedValueModel> variables;
    private ErrorModel error;
    private boolean isSuccessful = true;

    private Builder() {}

    public Builder withMessage(MessageModel message) {
      this.message = message;
      return this;
    }

    public Builder withVariables(Map<String, TypedValueModel> variables) {
      this.variables = variables;
      return this;
    }

    public Builder withError(ErrorModel error) {
      return withError(error, false);
    }

    public Builder withError(ErrorModel error, boolean markSuccessfull) {
      this.error = error;
      this.isSuccessful = !(markSuccessfull && error != null);

      return this;
    }

    public EventModel build() {
      return new EventModel(this.message, this.variables, this.error, this.isSuccessful);
    }
  }


}
