/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.event.model;

import static java.util.Objects.requireNonNull;

import java.util.Arrays;

/**
 * * Maintains a value that has an associated {@link DataTypeModel}.
 *
 * @since 1.0
 */
public class TypedValueModel {

  private DataTypeModel dataType;
  private byte[] content;
  private boolean truncated;

  // Used by json serializer/deserialzer
  public TypedValueModel() {}

  private TypedValueModel(DataTypeModel dataType, byte[] content, boolean truncated) {
    requireNonNull(dataType, "dataType cannot be null");

    this.dataType = dataType;
    this.content = content;
    this.truncated = truncated;
  }

  /**
   * @return the {@link DataTypeModel} associated
   */
  public DataTypeModel getDataType() {
    return dataType;
  }

  /**
   * @return this object's content
   */
  public byte[] getContent() {
    return content;
  }

  /**
   * @return whether it's content is truncated or not.
   */
  public boolean isTruncated() {
    return truncated;
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }
    if (o == null || getClass() != o.getClass()) {
      return false;
    }

    TypedValueModel that = (TypedValueModel) o;

    if (truncated != that.truncated) {
      return false;
    }
    if (!dataType.equals(that.dataType)) {
      return false;
    }
    return Arrays.equals(content, that.content);
  }

  /**
   * @return a {@link TypedValueModel.Builder} to build a new {@link TypedValueModel}.
   */
  public static Builder builder() {
    return new Builder();
  }

  @Override
  public int hashCode() {
    int result = dataType.hashCode();
    result = 31 * result + Arrays.hashCode(content);
    result = 31 * result + (truncated ? 1 : 0);
    return result;
  }

  public static class Builder {

    private DataTypeModel dataType;
    private byte[] content;
    private boolean truncated = false;

    private Builder() {}

    public Builder withDataType(DataTypeModel dataTypeModel) {
      this.dataType = dataTypeModel;
      return this;
    }

    public Builder withContent(byte[] content) {
      this.content = content;
      return this;
    }

    public Builder withTruncated(boolean truncated) {
      this.truncated = truncated;
      return this;
    }

    public TypedValueModel build() {
      return new TypedValueModel(this.dataType, this.content, this.truncated);
    }

  }

}
