/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.event.model;

import org.junit.Before;
import org.junit.Test;
import static org.junit.Assert.*;
import java.util.HashMap;
import java.util.Map;

public class EventModelTest {

  private EventModel eventModel;
  private MessageModel messageModel;
  private ErrorModel errorModel;
  private Map<String, TypedValueModel> variables;

  @Before
  public void setUp() {
    messageModel = new MessageModel(); // Assuming a default constructor exists
    errorModel = new ErrorModel(); // Assuming a default constructor exists
    variables = new HashMap<>();

    eventModel = EventModel.builder()
        .withMessage(messageModel)
        .withVariables(variables)
        .withError(errorModel)
        .build();
  }

  @Test
  public void testGetMessage() {
    assertEquals(messageModel, eventModel.getMessage());
  }

  @Test
  public void testGetVariables() {
    assertEquals(variables, eventModel.getVariables());
  }

  @Test
  public void testGetError() {
    assertEquals(errorModel, eventModel.getError());
  }

  @Test
  public void testIsSuccessful() {
    assertTrue(eventModel.isSuccessful());
  }

  @Test
  public void testSetSuccessful() {
    eventModel.setSuccessful(false);
    assertFalse(eventModel.isSuccessful());
  }

  @Test
  public void t1estEqualsAndHashCode() {
    EventModel anotherEventModel = EventModel.builder()
        .withMessage(messageModel)
        .withVariables(variables)
        .withError(errorModel)
        .build();

    assertEquals(eventModel, anotherEventModel);

  }

  @Test
  public void testBuilderWithErrorAndMarkSuccessful() {
    EventModel eventWithError = EventModel.builder()
        .withMessage(messageModel)
        .withVariables(variables)
        .withError(errorModel, true)
        .build();

    assertFalse(eventWithError.isSuccessful());
  }

  @Test
  public void testEqualsAndHashCode() {
    assertFalse(eventModel.equals(null));
    assertTrue(eventModel.equals(eventModel));
    EventModel sameEventModel = EventModel.builder()
        .withMessage(messageModel)
        .withVariables(variables)
        .withError(errorModel)
        .build();

    assertEquals(eventModel, sameEventModel);
    assertEquals(eventModel.hashCode(), sameEventModel.hashCode());


    // Different message
    MessageModel differentMessage = MessageModel.builder().withPayload(new TypedValueModel()).build();// assuming not equal
    EventModel withDifferentMessage = EventModel.builder()
        .withMessage(differentMessage)
        .withVariables(variables)
        .withError(errorModel)
        .build();
    assertFalse(eventModel.equals(withDifferentMessage));


    // Different variables
    Map<String, TypedValueModel> differentVariables = new HashMap<>();
    differentVariables.put("key", new TypedValueModel());
    EventModel withDifferentVariables = EventModel.builder()
        .withMessage(messageModel)
        .withVariables(differentVariables)
        .withError(errorModel)
        .build();
    assertFalse(eventModel.equals(withDifferentVariables));

    // Different error
    ErrorModel differentError = new ErrorModel(); // assuming not equal
    EventModel withDifferentError = EventModel.builder()
        .withMessage(messageModel)
        .withVariables(variables)
        .withError(differentError)
        .build();
    assertTrue(eventModel.equals(withDifferentError));


  }

}
