/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.event.model;

import org.junit.Before;
import org.junit.Test;
import static org.junit.Assert.*;

public class MessageModelTest {

  private MessageModel messageModel;
  private TypedValueModel attributes;
  private TypedValueModel payload;

  @Before
  public void setUp() {
    // Use the builder to create TypedValueModel instances
    DataTypeModel dataType = new DataTypeModel(); // Assuming a default constructor exists
    byte[] data = new byte[0]; // Example byte array, replace with actual data if needed

    attributes = TypedValueModel.builder()
        .withDataType(dataType)
        .withContent(data)
        .withTruncated(false)
        .build();

    payload = TypedValueModel.builder()
        .withDataType(dataType)
        .withContent(data)
        .withTruncated(false)
        .build();

    messageModel = MessageModel.builder()
        .withAttributes(attributes)
        .withPayload(payload)
        .build();
  }

  @Test
  public void testGetAttributes() {
    assertEquals(attributes, messageModel.getAttributes());
  }

  @Test
  public void testGetPayload() {
    assertEquals(payload, messageModel.getPayload());
  }

  @Test
  public void testEqualsAndHashCode() {
    MessageModel anotherMessageModel = MessageModel.builder()
        .withAttributes(attributes)
        .withPayload(payload)
        .build();

    assertEquals(messageModel, anotherMessageModel);
  }

  @Test
  public void testBuilderWithDifferentAttributes() {
    TypedValueModel differentAttributes = TypedValueModel.builder()
        .withDataType(new DataTypeModel())
        .withContent(new byte[] {1, 2, 3})
        .withTruncated(true)
        .build();

    MessageModel messageWithDifferentAttributes = MessageModel.builder()
        .withAttributes(differentAttributes)
        .withPayload(payload)
        .build();

    assertFalse(messageModel.equals(messageWithDifferentAttributes));
  }

  @Test
  public void testBuilderWithDifferentPayload() {
    TypedValueModel differentPayload = TypedValueModel.builder()
        .withDataType(new DataTypeModel())
        .withContent(new byte[] {4, 5, 6})
        .withTruncated(true)
        .build();

    MessageModel messageWithDifferentPayload = MessageModel.builder()
        .withAttributes(attributes)
        .withPayload(differentPayload)
        .build();

    assertFalse(messageModel.equals(messageWithDifferentPayload));
  }

  @Test
  public void testEqualsWithNull() {
    assertFalse(messageModel.equals(null));
  }

  @Test
  public void testEqualsWithSameObject() {
    assertTrue(messageModel.equals(messageModel));
  }

  @Test
  public void testHashCodeConsistency() {
    int initialHashCode = messageModel.hashCode();
    assertEquals(initialHashCode, messageModel.hashCode());
  }
}
