/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.event.model.component.location;

import org.junit.Before;
import org.junit.Test;
import static org.junit.Assert.*;
import org.mule.tooling.event.model.component.TypedComponentIdentifier;
import java.util.ArrayList;
import java.util.List;

public class ComponentLocationTest {

  private ComponentLocation componentLocation;
  private TypedComponentIdentifier identifier;
  private List<LocationPart> parts;

  @Before
  public void setUp() {
    identifier = new TypedComponentIdentifier();
    identifier.setType("componentType");
    identifier.setNamespace("componentNamespace");
    identifier.setName("componentName");

    LocationPart part = new LocationPart();
    part.setPartPath("/path/to/part");
    part.setFileName("fileName.xml");
    part.setLineInFile(42);
    part.setTypedComponentIdentifier(identifier);

    parts = new ArrayList<>();
    parts.add(part);

    componentLocation = new ComponentLocation();
    componentLocation.setLocation("/location");
    componentLocation.setFileName("fileName.xml");
    componentLocation.setLineInFile(42);
    componentLocation.setParts(parts);
    componentLocation.setTypedComponentIdentifier(identifier);
  }

  @Test
  public void testGetLocation() {
    assertEquals("/location", componentLocation.getLocation());
  }

  @Test
  public void testGetFileName() {
    assertEquals("fileName.xml", componentLocation.getFileName());
  }

  @Test
  public void testGetLineInFile() {
    assertEquals(Integer.valueOf(42), componentLocation.getLineInFile());
  }

  @Test
  public void testGetParts() {
    assertEquals(parts, componentLocation.getParts());
  }

  @Test
  public void testGetTypedComponentIdentifier() {
    assertEquals(identifier, componentLocation.getTypedComponentIdentifier());
  }

  @Test
  public void testEqualsAndHashCode() {
    ComponentLocation anotherComponentLocation = new ComponentLocation();
    anotherComponentLocation.setLocation("/location");
    anotherComponentLocation.setFileName("fileName.xml");
    anotherComponentLocation.setLineInFile(42);
    anotherComponentLocation.setParts(parts);
    anotherComponentLocation.setTypedComponentIdentifier(identifier);

    assertEquals(componentLocation, anotherComponentLocation);
    assertEquals(componentLocation.hashCode(), anotherComponentLocation.hashCode());
  }

  @Test
  public void testComponentLocationBuilder() {
    ComponentLocation builtComponentLocation = new ComponentLocation.ComponentLocationBuilder()
        .withLocation("/location")
        .withFileName("fileName.xml")
        .withLineInFile(42)
        .withParts(parts)
        .withTypedComponentIdentifier(identifier)
        .build();

    assertEquals(componentLocation, builtComponentLocation);
  }

  @Test
  public void testLocationPartBuilder() {
    LocationPart builtLocationPart = new LocationPart.LocationPartBuilder()
        .withPartPath("/path/to/part")
        .withFileName("fileName.xml")
        .withLineInFile(42)
        .withTypedComponentIdentifier(identifier)
        .build();

    assertEquals(parts.get(0), builtLocationPart);
  }
}
