/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.event.model;

import static java.util.Objects.requireNonNull;
import static org.mule.tooling.event.model.ModelType.collectionModelType;
import static org.mule.tooling.event.model.ModelType.simpleModelType;

/**
 * Defines a Java type and its association with additional information about the data, like MIME type and encoding.
 *
 * @since 1.0
 */
public class DataTypeModel {

  private String type;
  private String mediaType;
  @Deprecated
  private ModelType modelType;

  // Used by json serializer/deserializer
  public DataTypeModel() {}

  protected DataTypeModel(String type, String mediaType, ModelType modelType) {
    requireNonNull(type, "type cannot be null");
    requireNonNull(mediaType, "mediaType cannot be null");

    this.type = type;
    this.mediaType = mediaType;
    this.modelType = modelType;
  }

  /**
   * @return the String representation of the media type.
   */
  public String getMediaType() {
    return mediaType;
  }

  /**
   * @return the class object of the source object, represented as String.
   */
  public String getType() {
    return type;
  }

  /**
   * @return {@link ModelType} of the Data Type
   */
  public ModelType getModelType() {
    return modelType;
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }
    if (o == null || getClass() != o.getClass()) {
      return false;
    }

    DataTypeModel that = (DataTypeModel) o;

    if (!type.equals(that.type)) {
      return false;
    }
    if (!modelType.equals(that.modelType)) {
      return false;
    }
    return mediaType.equals(that.mediaType);
  }

  @Override
  public int hashCode() {
    int result = type.hashCode();
    result = 31 * result + mediaType.hashCode();
    return result;
  }

  public static Builder builder() {
    return new Builder();
  }

  public static class Builder {

    private String type;
    private String mediaType;
    private ModelType modelType = simpleModelType();

    private Builder() {}

    public Builder withType(String type) {
      this.type = type;
      return this;
    }

    public Builder withMediaType(String mediaType) {
      this.mediaType = mediaType;
      return this;
    }

    public Builder asCollectionWithItemsType(String itemsType) {
      modelType = collectionModelType(itemsType);
      return this;
    }

    public DataTypeModel build() {
      return new DataTypeModel(this.type, this.mediaType, modelType);
    }

  }

}
