/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.event.model.component.location;

import org.mule.tooling.event.model.component.TypedComponentIdentifier;

import java.util.Objects;

/**
 * Representation a location part of a {@link ComponentLocation} in a Mule Configuration.
 *
 * @since 1.0
 */
public class LocationPart {

  private String partPath;
  private String fileName;
  private Integer lineInFile;
  private TypedComponentIdentifier typedComponentIdentifier;

  // Used by json serializer/deserialzer
  public LocationPart() {}

  public String getPartPath() {
    return this.partPath;
  }

  public void setPartPath(String partPath) {
    this.partPath = partPath;
  }

  public TypedComponentIdentifier getPartIdentifier() {
    return this.typedComponentIdentifier;
  }

  public void setPartIdentifier(TypedComponentIdentifier typedComponentIdentifier) {
    this.typedComponentIdentifier = typedComponentIdentifier;
  }

  public String getFileName() {
    return this.fileName;
  }

  public void setFileName(String fileName) {
    this.fileName = fileName;
  }

  public Integer getLineInFile() {
    return this.lineInFile;
  }

  public void setLineInFile(Integer lineInFile) {
    this.lineInFile = lineInFile;
  }

  public TypedComponentIdentifier getTypedComponentIdentifier() {
    return this.typedComponentIdentifier;
  }

  public void setTypedComponentIdentifier(TypedComponentIdentifier typedComponentIdentifier) {
    this.typedComponentIdentifier = typedComponentIdentifier;
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }
    if (o == null || getClass() != o.getClass()) {
      return false;
    }

    LocationPart that = (LocationPart) o;

    if (!Objects.equals(getPartPath(), that.getPartPath())) {
      return false;
    }
    if (getPartIdentifier() != null ? !getPartIdentifier().equals(that.getPartIdentifier())
        : that.getPartIdentifier() != null) {
      return false;
    }
    if (getFileName() != null ? !getFileName().equals(that.getFileName()) : that.getFileName() != null) {
      return false;
    }
    return getLineInFile() != null ? getLineInFile().equals(that.getLineInFile()) : that.getLineInFile() == null;
  }

  @Override
  public int hashCode() {
    int result = getPartPath() != null ? getPartPath().hashCode() : 31;
    result = 31 * result + (getPartIdentifier() != null ? getPartIdentifier().hashCode() : 0);
    result = 31 * result + (getFileName() != null ? getFileName().hashCode() : 0);
    result = 31 * result + (getLineInFile() != null ? getLineInFile().hashCode() : 0);
    return result;
  }

  @Override
  public String toString() {
    return "DefaultLocationPart{" +
        "partPath='" + partPath + '\'' +
        ", partIdentifier=" + typedComponentIdentifier +
        ", fileName='" + fileName + '\'' +
        ", lineInFile=" + lineInFile +
        '}';
  }

  public static class LocationPartBuilder {

    private LocationPart locationPart;

    public LocationPartBuilder() {
      this.locationPart = new LocationPart();
    }

    public LocationPartBuilder withPartPath(String partPath) {
      this.locationPart.setPartPath(partPath);
      return this;
    }

    public LocationPartBuilder withFileName(String fileName) {
      this.locationPart.setFileName(fileName);
      return this;
    }

    public LocationPartBuilder withLineInFile(Integer lineInFile) {
      this.locationPart.setLineInFile(lineInFile);
      return this;
    }

    public LocationPartBuilder withTypedComponentIdentifier(TypedComponentIdentifier typedComponentIdentifier) {
      this.locationPart.setTypedComponentIdentifier(typedComponentIdentifier);
      return this;
    }

    public LocationPart build() {
      return this.locationPart;
    }

  }

}
