/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.runtime.process.controller;

import static java.lang.Integer.parseInt;
import static java.util.regex.Pattern.compile;
import org.mule.tooling.runtime.process.controller.command.StatusMuleCommand;

import java.io.File;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Unix specific process controller for Mule Runtime.
 *
 * @since 4.0
 */
class UnixMuleProcessController extends AbstractMuleProcessController {

  private static final String MULE_UNIX_BINARY = "mule";
  private static final String STATUS =
      "Mule(\\sEnterprise Edition \\(standalone\\))? is running(\\s|:\\sPID:)?\\(?([0-9]+)\\)?.*";
  private static final Pattern STATUS_PATTERN = compile(STATUS);
  private static final String STATUS_414 =
      "Mule(\\sEnterprise Edition \\(not installed\\))? is running(\\s|:\\sPID:)?\\(?([0-9]+)\\)?.*";
  private static final Pattern STATUS_414_PATTERN = compile(STATUS_414);


  private static final String STATUS_FALLBACK = "Mule Enterprise Edition is running \\(([0-9]+)\\)\\.";
  private static final Pattern STATUS_FALLBAC_PATTERN = compile(STATUS_FALLBACK);

  /**
   * Creates an instance of a Unix process controller for Mule Runtime.
   *
   * @param muleHome {@link File} location of the local Mule Runtime {@code MULE_HOME}
   * @param timeout  milliseconds to wait for Mule Runtime operations to be completed.
   */
  UnixMuleProcessController(File muleHome, long timeout) {
    super(muleHome, timeout, MULE_UNIX_BINARY);
  }

  @Override
  protected int getPID() throws Exception {
    StatusMuleCommand command = new StatusMuleCommand(this.getMuleServiceWrapper());
    command.execute();

    final String output = command.getProcessOutput().getString();
    Matcher matcher = STATUS_PATTERN.matcher(output);
    Matcher fallback414Matcher = STATUS_414_PATTERN.matcher(output);
    Matcher fallbackMatcher = STATUS_FALLBAC_PATTERN.matcher(output);
    if (matcher.find()) {
      return parseInt(matcher.group(3));
    } else if (fallback414Matcher.find()) {
      return parseInt(fallback414Matcher.group(3));
    } else if (fallbackMatcher.find()) {
      return parseInt(fallbackMatcher.group(1));
    } else {
      return -1;
    }
  }

}
