/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.runtime.process.controller;

import static java.lang.Runtime.getRuntime;
import static java.util.concurrent.TimeUnit.MILLISECONDS;
import static org.awaitility.Awaitility.with;
import static org.zeroturnaround.process.ProcessUtil.destroyGracefullyOrForcefullyAndWait;
import org.mule.tooling.runtime.process.controller.command.ConsoleMuleCommand;
import org.mule.tooling.runtime.process.controller.exception.MuleProcessException;

import java.io.File;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.zeroturnaround.exec.StartedProcess;
import org.zeroturnaround.process.PidProcess;
import org.zeroturnaround.process.Processes;

/**
 * Base implementation for {@link MuleProcessController}.
 *
 * @since 4.0
 */
public abstract class AbstractMuleProcessController implements MuleProcessController {

  private final Logger logger = LoggerFactory.getLogger(this.getClass());

  private final MuleServiceWrapper muleServiceWrapper;
  private StartedProcess consoleProcess;
  private PidProcess wrapperProcess;
  private Thread shutdownHook;

  /**
   * Creates an instance of a controller for Mule Runtime.
   *
   * @param muleHome   {@link File} location of the local Mule Runtime {@code MULE_HOME}
   * @param timeout    milliseconds to wait for Mule Runtime operations to be completed.
   * @param muleBinary binary script to be called in order to execute commands.
   */
  AbstractMuleProcessController(File muleHome, long timeout, String muleBinary) {
    final File bin = new File(new File(muleHome, "bin"), muleBinary);
    checkState(bin.exists(), muleBinary + " not present");
    checkState(bin.canExecute(), muleBinary + " cannot be executed");

    muleServiceWrapper = new MuleServiceWrapper(muleHome, timeout, muleBinary);
  }

  private void checkState(boolean condition, String errorMessage) {
    if (!condition) {
      throw new IllegalStateException(errorMessage);
    }
  }

  protected MuleServiceWrapper getMuleServiceWrapper() {
    return this.muleServiceWrapper;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void start(String[] args) throws MuleProcessException {
    if (logger.isInfoEnabled()) {
      logger.info("Starting Mule Runtime from: {}", muleServiceWrapper.getMuleHome().getAbsolutePath());
    }
    // Register shutdown hook to stop Mule
    shutdownHook = new Thread(() -> AbstractMuleProcessController.this.stop());
    getRuntime().addShutdownHook(shutdownHook);

    ConsoleMuleCommand consoleCommand = new ConsoleMuleCommand(this.muleServiceWrapper, args);
    consoleCommand.execute();
    consoleProcess = consoleCommand.getProcess();


    // Waits until the service wrapper process is started...
    with().timeout(this.muleServiceWrapper.getProcessInvocationTimeout(), MILLISECONDS)
        .pollInterval(500, MILLISECONDS).pollDelay(250, MILLISECONDS).until(() -> getPID() != -1);

    int pid;
    try {
      pid = getPID();
    } catch (Exception e) {
      stopQuietly();
      throw new MuleProcessException("Error while getting PID for wrapper process", e);
    }

    if (pid == -1) {
      stopQuietly();
      throw new MuleProcessException("Could not get PID for wrapper");
    }

    wrapperProcess = Processes.newPidProcess(pid);
    if (logger.isDebugEnabled()) {
      logger.debug("Wrapper process PID: {}", wrapperProcess.getPid());
    }
  }

  protected abstract int getPID() throws Exception;

  private void stopQuietly() {
    try {
      this.stop();
    } catch (Exception e) {
      // Nothing to do...
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isRunning() throws MuleProcessException {
    if (logger.isDebugEnabled()) {
      logger.debug("Checking if Mule Runtime is running");
    }
    try {
      return wrapperProcess != null ? wrapperProcess.isAlive() : false;
    } catch (Exception e) {
      if (logger.isWarnEnabled()) {
        logger.warn("Error while checking status", e);
      }
      return false;
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void stop() throws MuleProcessException {
    if (logger.isInfoEnabled()) {
      logger.info("Stopping Mule Runtime");
    }
    if (wrapperProcess != null) {
      try {
        if (logger.isDebugEnabled()) {
          logger.debug("Destroying service wrapper process");
        }
        destroyGracefullyOrForcefullyAndWait(wrapperProcess);
        getRuntime().removeShutdownHook(shutdownHook);
      } catch (Exception e) {
        throw new MuleProcessException(e);
      } finally {
        // console process has to be destroyed too.
        consoleProcess.getProcess().destroy();
      }
    }
  }

}
