/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.runtime.process.controller.command;

import static java.lang.String.join;
import static java.util.Arrays.asList;
import static java.util.Collections.singletonList;
import static java.util.Objects.requireNonNull;
import static java.util.concurrent.TimeUnit.MILLISECONDS;
import static org.mule.tooling.runtime.process.controller.MuleServiceWrapper.MULE_HOME;
import org.mule.tooling.runtime.process.controller.MuleServiceWrapper;
import org.mule.tooling.runtime.process.controller.exception.MuleProcessException;

import java.util.ArrayList;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.zeroturnaround.exec.ProcessExecutor;
import org.zeroturnaround.exec.stream.slf4j.Slf4jStream;

/**
 * Base class for commands.
 *
 * @since 4.0
 */
public abstract class AbstractMuleCommand implements MuleCommand {

  protected final Logger logger = LoggerFactory.getLogger(this.getClass());
  private final Logger runtimeLogger = LoggerFactory.getLogger("com.mulesoft.mule.distributions");

  private MuleServiceWrapper muleServiceWrapper;

  /**
   * Creates an instance of the command.
   *
   * @param muleServiceWrapper {@link MuleServiceWrapper} to execute the command. Non null.
   */
  public AbstractMuleCommand(MuleServiceWrapper muleServiceWrapper) {
    requireNonNull(muleServiceWrapper, "muleServiceWrapper cannot be null");
    this.muleServiceWrapper = muleServiceWrapper;
  }

  /**
   * @return {@link MuleServiceWrapper} to allow execution of commands.
   */
  protected MuleServiceWrapper getMuleServiceWrapper() {
    return this.muleServiceWrapper;
  }

  /**
   * @param command string with the command to be executed
   * @return a {@link ProcessExecutor} with defaults settings, {@value MuleServiceWrapper#MULE_HOME}, timeout and logger.
   */
  protected ProcessExecutor createMuleProcessExecutor(String command) {
    return this.createMuleProcessExecutor(singletonList(command));
  }

  /**
   * @param command string with the command to be executed
   * @param args    string list containing the arguments
   * @return a {@link ProcessExecutor} with defaults settings, {@value MuleServiceWrapper#MULE_HOME}, timeout and logger.
   */
  protected ProcessExecutor createMuleProcessExecutor(String command, String[] args) {
    List<String> commands = new ArrayList<>();
    commands.add(command);
    commands.addAll(asList(args));
    return this.createMuleProcessExecutor(commands);
  }

  /**
   * @param command string list containing the program and its arguments
   * @return a {@link ProcessExecutor} with defaults settings, {@value MuleServiceWrapper#MULE_HOME}, timeout and logger.
   */
  protected ProcessExecutor createMuleProcessExecutor(List<String> command) {
    List<String> commands = new ArrayList<>();
    commands.add(this.getMuleServiceWrapper().getMuleBinaryCommand());
    commands.addAll(command);

    return new ProcessExecutor()
        .commandSplit(join(" ", commands))
        .redirectError(Slf4jStream.of(runtimeLogger).asError())
        .redirectOutput(Slf4jStream.of(runtimeLogger).asDebug())
        .timeout(getMuleServiceWrapper().getProcessInvocationTimeout(), MILLISECONDS)
        .environment(MULE_HOME, getMuleServiceWrapper().getMuleHome().getAbsolutePath());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final void execute() {
    try {
      this.doExecute();
    } catch (Exception e) {
      throw new MuleProcessException("Error while executing command", e);
    }
  }

  protected static void checkState(boolean condition, String errorMessage) {
    if (!condition) {
      throw new IllegalStateException(errorMessage);
    }
  }

  /**
   * Template method for commands.
   */
  protected abstract void doExecute() throws Exception;
}
