/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.api.el;

import java.util.List;
import java.util.Optional;

import static java.util.Collections.emptyList;
import static java.util.Collections.unmodifiableList;

/**
 * Represents the result of an expression validation.
 *
 * @since 1.0
 */
public class ValidationResult {

  private boolean result;
  private String errorMessage;
  private List<ValidationMessage> messages;

  public static ValidationResult success() {
    return new ValidationResult(true, null);
  }

  public static ValidationResult failure(String message) {
    return new ValidationResult(false, message);
  }

  public static ValidationResult failure(String message, List<ValidationMessage> messages) {
    return new ValidationResult(false, message, messages);
  }

  public static ValidationResult failure(String message, String expression) {
    return failure(String.format("%s. Offending expression string is: %s", message, expression));
  }

  public ValidationResult() {}

  public ValidationResult(boolean result, String errorMessage) {
    this(result, errorMessage, emptyList());
  }

  public ValidationResult(boolean result, String errorMessage, List<ValidationMessage> messages) {
    this.result = result;
    this.errorMessage = errorMessage;
    this.messages = unmodifiableList(messages);
  }

  /**
   * @return an optional representing the generic validation error or an empty one
   */
  public Optional<String> errorMessage() {
    return Optional.ofNullable(errorMessage);
  }

  /**
   * @return a list of all {@link ValidationMessage}s found or an empty list if no relevant data is present
   */
  public List<ValidationMessage> messages() {
    return messages;
  }

  /**
   * @return true if the validation was ok, false otherwise
   */
  public boolean isSuccess() {
    return result;
  }

}
