/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.api.extension.model;

import static java.util.Optional.ofNullable;

import java.util.Optional;

/**
 * Describes a library that the extension depends on but is not packaged with it.
 *
 * @since 1.0
 */
public class ExternalLibraryModel {

  private String name;
  private String description;
  private String requiredClassName;
  private String regexMatcher;
  private ExternalLibraryType type;
  private String suggestedCoordinates;
  private boolean optional;

  // Just needed in order to serialize this object
  private ExternalLibraryModel() {}

  public ExternalLibraryModel(String name, String description, String requiredClassName, String regexMatcher,
                              ExternalLibraryType type, String suggestedCoordinates, boolean optional) {
    this.name = name;
    this.description = description;
    this.requiredClassName = requiredClassName;
    this.regexMatcher = regexMatcher;
    this.type = type;
    this.suggestedCoordinates = suggestedCoordinates;
    this.optional = optional;
  }

  /**
   * @return The library's name. Will never be {@code null}
   */
  public String getName() {
    return name;
  }

  /**
   * @return The library's description
   */
  public String getDescription() {
    return description;
  }

  /**
   * If present, returns a regexp which must match the name of the library's file.
   *
   * @return An {@link Optional} regexp to match the library's file name
   */
  public Optional<String> getRegexMatcher() {
    return ofNullable(regexMatcher);
  }

  /**
   * If present, the library should contain a class of the given name
   *
   * @return The name of the required class
   */
  public Optional<String> getRequiredClassName() {
    return ofNullable(requiredClassName);
  }

  /**
   * @return The library's type
   */
  public ExternalLibraryType getType() {
    return type;
  }

  /**
   * If provided, suggests Maven coordinates where the required library can be found. This coordinates should
   * follow the Maven convention: {@code groupId:artifactId:packaging:classifier:version}.
   * <p>
   * Keep in mind that not all the values of the coordinates are required, for example:
   * {@code org.mule.modules:a-required-lib:1.0.0} are valid coordinates, which communicates the {@code groupId},
   * {@code artifactId} and {@code version} of the external library.
   * <p>
   * By default, the packaging is {@code jar}, so if is required to use a native library, like a .DLL, you will provide:
   * {@code org.mule.module:a-native-lib:dll:1.0.0} where {@code dll} is the packaging of the library.
   * <p>
   * More information in: <a href="https://maven.apache.org/pom.html#Maven_Coordinates">https://maven.apache.org/pom.html#Maven_Coordinates</a>
   *
   * @return The optional maven coordinates.
   */
  public Optional<String> getSuggestedCoordinates() {
    return ofNullable(suggestedCoordinates);
  }

  /**
   * @return A boolean indicating whether the library is required or not to the basic functioning of the connector.
   */
  public boolean isOptional() {
    return optional;
  }
}
