/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.api;

import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.runtime.api.app.declaration.ArtifactDeclaration;
import org.mule.tooling.client.api.artifact.ToolingArtifact;
import org.mule.tooling.client.api.artifact.declaration.ArtifactSerializationService;
import org.mule.tooling.client.api.configuration.agent.AgentConfiguration;
import org.mule.tooling.client.api.exception.ToolingArtifactNotFoundException;
import org.mule.tooling.client.api.extension.ExtensionModelService;
import org.mule.tooling.client.api.tryit.TryItService;

import java.io.InputStream;
import java.net.URL;

/**
 * Provides tooling services for Mule Runtime.
 *
 * @since 1.0
 */
public interface ToolingRuntimeClient {

  interface Builder {

    /**
     * Sets the {@link AgentConfiguration} used by the client to access the Tooling REST API. If the operations executed by the
     * client don't require to access Tooling REST API this won't need to be set. Otherwise an operation could throw a
     * {@link org.mule.tooling.client.api.exception.MissingToolingConfigurationException} in case if there was no configuration
     * set for the client instance.
     *
     * @param agentConfiguration {@link AgentConfiguration} for the client.
     * @return this
     */
    Builder withRemoteAgentConfiguration(AgentConfiguration agentConfiguration);

    /**
     * Sets the {@link MavenConfiguration} used by the client to access the maven artifacts.
     *
     * If not provided, then the bootstrap maven configuration will be used.
     *
     * @param mavenConfiguration {@link MavenConfiguration} for the client.
     * @return this
     */
    Builder withMavenConfiguration(MavenConfiguration mavenConfiguration);


    /**
     * Sets the logging configuration file used by the Mule logging mechanism when deploying artifacts.
     * This configures Mule Agent, so different Mule Tooling Clients using the same Mule Agent will share
     * this configuration.
     *
     * If provided, it will use the last configuration provided. 
     * If not provided, then the default configuration will be used.
     *
     * @param loggingConfiguration the logging configuration file to be used.
     * @return this
     */
    Builder withLoggingConfiguration(InputStream loggingConfiguration);

    /**
     * Builds the {@link ToolingRuntimeClient}.
     *
     * @return a new instance of the {@link ToolingRuntimeClient}.
     */
    ToolingRuntimeClient build();

  }

  /**
   * Creates a {@link ToolingArtifact} for the application from the URL of the application content. It can be a file protocol
   * pointing to a the expanded content of the application jar or a remote location pointing to the jar file where it would be read.
   * <p/>
   * It is expected that users of this API should handle the packaging for the application content, therefore if {@link ToolingArtifact ToolingArtifacts}
   * are created per request keeping the same applicationUrlContent when there is no need to package again would avoid wasting time on
   * packaging when not needed.
   *
   * @param applicationUrlContent {@link URL} for the application content.
   * @return a {@link ToolingArtifact}.
   */
  ToolingArtifact newToolingArtifact(URL applicationUrlContent);

  /**
   * Fetches a {@link ToolingArtifact} already created to resolve operations over an application.
   *
   * @param id the identifier for the {@link ToolingArtifact} to be recovered.
   * @return a {@link ToolingArtifact}
   * @throws ToolingArtifactNotFoundException if the {@link ToolingArtifact} is no longer present.
   */
  ToolingArtifact fetchToolingArtifact(String id) throws ToolingArtifactNotFoundException;

  /**
   * Returns a {@link ExtensionModelService} to allow retrieving the {@link org.mule.runtime.api.meta.model.ExtensionModel} of a
   * plugin.
   *
   * @return a {@link ExtensionModelService}
   */
  ExtensionModelService extensionModelService();

  /**
   * Returns an {@link ArtifactSerializationService} that allows to serialize and deserialize an {@link ArtifactDeclaration}.
   *
   * @return a {@link ArtifactSerializationService}
   */
  ArtifactSerializationService artifactSerializationService();

  /**
   * Returns a {@link TryItService} to do try it over an application.
   *
   * @return a {@link TryItService}
   */
  TryItService tryItService();

}
