/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.api.extension.model;

import static java.util.Optional.ofNullable;
import org.mule.metadata.api.model.ObjectType;
import org.mule.tooling.client.api.extension.model.config.ConfigurationModel;
import org.mule.tooling.client.api.extension.model.connection.ConnectionProviderModel;
import org.mule.tooling.client.api.extension.model.construct.ConstructModel;
import org.mule.tooling.client.api.extension.model.function.FunctionModel;
import org.mule.tooling.client.api.extension.model.operation.OperationModel;
import org.mule.tooling.client.api.extension.model.source.SourceModel;

import java.util.List;
import java.util.Optional;
import java.util.Set;

import org.apache.commons.lang3.builder.ReflectionToStringBuilder;

/**
 * An Extension that provides packaged functionality.
 * <p>
 * Extensions can augment a system by providing new features in the form of operations and connection providers.
 * What makes the extension model different from a class implementing a certain interface is the fact that extensions
 * provide enough information at runtime so that they can be used without prior knowledge, and can be both executed
 * or integrated into tooling seamlessly.
 * <p>
 * An extension model is not a miscellaneous group of methods, but can be seen (and may be derived from) an object model.
 * As such, an extension will provide several ways to configure itself and
 * will provide a set of operations that may be eventually executed.
 * <p>
 * The extension model doesn't just map a JVM object model. Extensions provide richer metadata, and a dynamic
 * execution model, but more importantly they restrict the way operations are defined and used to a
 * manageable subset that would deterministic data flow analysis.
 * <p>
 * An extension doesn't define any predefined syntax, evaluation order or execution paradigm. The operations provided
 * are expected to be used as individual building blocks in a bigger system, hence the name <code>Extension</code>
 *
 * @since 1.0
 */
public class ExtensionModel {

  private String name;
  private Category category;
  private MuleVersion minMuleVersion;
  private String vendor;
  private Set<String> resources;
  private Set<ImportedTypeModel> importedTypes;
  private Set<SubTypesModel> subTypes;
  private Set<ObjectType> types;
  private String version;
  private XmlDslModel xmlDslModel;
  private String description;
  private DisplayModel displayModel;
  private Set<ExternalLibraryModel> externalLibraryModels;
  private List<ConfigurationModel> configurationModels;
  private List<OperationModel> operationModels;
  private List<FunctionModel> functionModels;
  private List<ConstructModel> constructModels;
  private List<ConnectionProviderModel> connectionProviders;
  private List<SourceModel> sourceModels;

  // Just needed in order to serialize this object
  private ExtensionModel() {}

  public ExtensionModel(String name, Category category, MuleVersion minMuleVersion, String vendor,
                        Set<String> resources,
                        Set<ImportedTypeModel> importedTypes,
                        Set<SubTypesModel> subTypes, Set<ObjectType> types, String version,
                        XmlDslModel xmlDslModel, String description,
                        DisplayModel displayModel,
                        Set<ExternalLibraryModel> externalLibraryModels,
                        List<ConfigurationModel> configurationModels,
                        List<OperationModel> operationModels,
                        List<ConnectionProviderModel> connectionProviders,
                        List<SourceModel> sourceModels,
                        List<FunctionModel> functionModels,
                        List<ConstructModel> constructModels) {
    this.name = name;
    this.category = category;
    this.minMuleVersion = minMuleVersion;
    this.vendor = vendor;
    this.resources = resources;
    this.importedTypes = importedTypes;
    this.subTypes = subTypes;
    this.types = types;
    this.version = version;
    this.xmlDslModel = xmlDslModel;
    this.description = description;
    this.displayModel = displayModel;
    this.externalLibraryModels = externalLibraryModels;
    this.configurationModels = configurationModels;
    this.operationModels = operationModels;
    this.connectionProviders = connectionProviders;
    this.sourceModels = sourceModels;
    this.functionModels = functionModels;
    this.constructModels = constructModels;
  }

  public String getName() {
    return name;
  }

  public String getDescription() {
    return description;
  }

  public Category getCategory() {
    return category;
  }

  public MuleVersion getMinMuleVersion() {
    return minMuleVersion;
  }

  public String getVendor() {
    return vendor;
  }

  public Set<String> getResources() {
    return resources;
  }

  public Set<ImportedTypeModel> getImportedTypes() {
    return importedTypes;
  }

  public Set<SubTypesModel> getSubTypes() {
    return subTypes;
  }

  public Set<ObjectType> getTypes() {
    return types;
  }

  public String getVersion() {
    return version;
  }

  public XmlDslModel getXmlDslModel() {
    return xmlDslModel;
  }

  public Optional<DisplayModel> getDisplayModel() {
    return ofNullable(displayModel);
  }

  public Set<ExternalLibraryModel> getExternalLibraryModels() {
    return externalLibraryModels;
  }

  public List<ConfigurationModel> getConfigurationModels() {
    return configurationModels;
  }

  public List<OperationModel> getOperationModels() {
    return operationModels;
  }

  public List<ConnectionProviderModel> getConnectionProviders() {
    return connectionProviders;
  }

  public List<SourceModel> getSourceModels() {
    return sourceModels;
  }

  public List<FunctionModel> getFunctionModels() {
    return functionModels;
  }

  public List<ConstructModel> getConstructModels() {
    return constructModels;
  }

  @Override
  public int hashCode() {
    return this.name.hashCode();
  }

  @Override
  public boolean equals(Object obj) {
    return this.getClass().isInstance(obj) && this.name.equals(((ExtensionModel) obj).getName());
  }

  @Override
  public String toString() {
    return ReflectionToStringBuilder.toString(this);
  }

}
