/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.api.location;


import static java.lang.String.join;
import static org.apache.commons.lang3.StringUtils.isNumeric;
import static org.mule.runtime.api.util.Preconditions.checkArgument;
import static org.mule.runtime.api.util.Preconditions.checkState;

import java.util.LinkedList;
import java.util.List;

/**
 * A location allows to define the position of a certain component in the configuration.
 * <p/>
 * Only components contained within global components with name can be referenced using a {@link Location} instance.
 * <p/>
 * The string representation of the implementation of this interface must be the serialized form of the location which consist of
 * the global element name and the parts separated by an slash character.
 *
 * @since 1.0
 */
public class Location {

  public static final String SOURCE = "source";
  public static final String CONNECTION = "connection";
  public static final String PARAMETERS = "parameters";
  public static final String PROCESSORS = "processors";
  public static final String ERROR_HANDLER = "errorHandler";

  protected static final String PARTS_SEPARATOR = "/";
  private LinkedList<String> parts = new LinkedList<>();

  public Location() {

  }

  public Location(List<String> parts) {
    this.parts.addAll(parts);
  }

  @Override
  public String toString() {
    return join(PARTS_SEPARATOR, parts);
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }
    if (o == null || getClass() != o.getClass()) {
      return false;
    }

    Location location = (Location) o;

    return parts.equals(location.parts);

  }

  @Override
  public int hashCode() {
    return parts.hashCode();
  }

  /**
   * @return the name of the global element that contains the component referenced by {@code this} {@link Location}.
   */
  public String getGlobalName() {
    return parts.get(0);
  }

  /**
   * @return the parts within the global element that define the location of the
   * component referenced by {@code this} {@link Location}.
   */
  public List<String> getParts() {
    return parts.subList(1, parts.size());
  }

  /**
   * @return a new builder instance.
   */
  public static Builder builder() {
    return new Builder();
  }

  /**
   * Creates a new {@link Builder} with the provided location represented as string as base.
   *
   * @param location a location to use to pre-configured the builder
   * @return a new builder instance with the provided location as base.
   */
  public static Builder builderFromStringRepresentation(String location) {
    String[] parts = location.split("/");
    Builder builder = Location.builder();
    builder = builder.globalName(parts[0]);
    for (int i = 1; i < parts.length; i++) {
      builder = builder.addPart(parts[i]);
    }
    return builder;
  }

  /**
   * A builder to create a {@link Location} object.
   * <p>
   * All {@link Location} instances must be created using this builder. The builder implementation may not be thread safe but it
   * is immutable so each method call in the builder returns a new instance so it can be reused.
   *
   * @since 1.0
   */
  public static class Builder {

    private Location location = new Location();
    private boolean globalNameAlreadySet = false;

    /**
     * Sets the name of the global component. This method must only be called once.
     *
     * @param globalName the name of the global component
     * @return a new builder with the provided configuration.
     */
    public Builder globalName(String globalName) {
      globalNameAlreadySet = true;
      verifyPartDoesNotContainsSlash(globalName);
      Builder Builder = builderCopy();
      Builder.location.parts.add(0, globalName);
      return Builder;
    }

    /**
     * Adds a new part at the end of the location.
     *
     * @param part the name of the part
     * @return a new builder with the provided configuration.
     */
    public Builder addPart(String part) {
      verifyPartDoesNotContainsSlash(part);
      verifyIndexPartAfterProcessor(part);
      Builder Builder = builderCopy();
      Builder.location.parts.addLast(part);
      return Builder;
    }

    /**
     * Adds a new {@link Location#CONNECTION} part at the end of the location.
     * <p>
     * Connection elements within a configuration component must be addressed using a {@link Location#CONNECTION} part.
     *
     * @return a new builder with the provided configuration.
     */
    public Builder addConnectionPart() {
      checkIsNotFirstPart(CONNECTION);
      Builder Builder = builderCopy();
      Builder.location.parts.add(CONNECTION);
      return Builder;
    }

    /**
     * Adds a new {@link Location#SOURCE} part at the end of the location.
     * <p>
     * Message sources within other component must be addressed using a {@link Location#SOURCE} part.
     *
     * @return a new builder with the provided configuration.
     */
    public Builder addSourcePart() {
      checkIsNotFirstPart(SOURCE);
      Builder Builder = builderCopy();
      Builder.location.parts.add(SOURCE);
      return Builder;
    }

    /**
     * Adds a new {@link Location#PROCESSORS} part at the end of the location.
     * <p>
     * Components that allow nested processors must have {@link Location#PROCESSORS} as part before the nested processors indexes.
     *
     * @return a new builder with the provided configuration.
     */
    public Builder addProcessorsPart() {
      checkIsNotFirstPart(PROCESSORS);
      Builder Builder = builderCopy();
      Builder.location.parts.add(PROCESSORS);
      return Builder;
    }

    /**
     * Adds a new {@link Location#ERROR_HANDLER} part at the end of the location.
     * <p>
     * Components that allow nested {@code on-error} components must have {@link Location#ERROR_HANDLER}
     * as part before the {@code on-error} indexes.
     *
     * @return a new builder with the provided configuration.
     */
    public Builder addErrorHandlerPart() {
      checkIsNotFirstPart(ERROR_HANDLER);
      Builder Builder = builderCopy();
      Builder.location.parts.add(ERROR_HANDLER);
      return Builder;
    }

    /**
     * Adds a new {@link Location#PARAMETERS} part at the end of the location.
     * <p>
     * Components that allow nested parameters must have {@link Location#PARAMETERS}
     * as part before the {@code parameter} name.
     *
     * @return a new builder with the provided configuration.
     */
    public Builder addParameterPart() {
      checkIsNotFirstPart(PARAMETERS);
      Builder Builder = builderCopy();
      Builder.location.parts.add(PARAMETERS);
      return Builder;
    }

    /**
     * Adds a new index part. The index part is used to reference a component within a collection.
     * <p>
     * There cannot be two index parts consecutively.
     *
     * @param index the index of the component.
     * @return a new builder with the provided configuration.
     */
    public Builder addIndexPart(int index) {
      checkState(!location.parts.isEmpty(), "An index cannot be the first part");
      Builder Builder = builderCopy();
      Builder.location.parts.addLast(String.valueOf(index));
      return Builder;
    }

    /**
     * Adds the parts of this location.
     *
     * @param parts the parts of the location
     * @return a new builder with the provided configuration.
     */
    public Builder parts(List<String> parts) {
      Builder Builder = builderCopy();
      Builder.location.parts = new LinkedList<>(parts);
      return Builder;
    }

    /**
     * @return a location built with the provided configuration.
     */
    public Location build() {
      checkState(globalNameAlreadySet, "global component name must be set");
      return location;
    }


    private void checkIsNotFirstPart(String partName) {
      checkState(!location.parts.isEmpty(), "[" + partName + "] cannot be the first part");
    }

    private void verifyIndexPartAfterProcessor(String part) {
      if (location.parts.getLast().equals(PROCESSORS)) {
        checkState(isNumeric(part), "Only an index part can follow a processors part");
      }
    }

    private Builder builderCopy() {
      Builder Builder = new Builder();
      Builder.globalNameAlreadySet = this.globalNameAlreadySet;
      Builder.location.parts.addAll(this.location.parts);
      return Builder;
    }

    private void verifyPartDoesNotContainsSlash(String globalName) {
      checkArgument(!globalName.contains(PARTS_SEPARATOR), "Slash cannot be part of the global name or part");
    }

  }

}

