/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.api.types;

import java.lang.reflect.InvocationTargetException;
import java.util.Arrays;

/**
 * Base class for representing enumerations from Mule Runtime API.
 * In case if a value is not matched it will set as {@link #isUnknown}. This could happen whenever a version of this API
 * is working against a Runtime version that added a new kind of type and cannot be mapped to the ones existing in this API, for
 * those cases (if needed the client has to be updated to use the latest version of this API) or pick up directly the value if no
 * logic is needed when handling the type represented by this class.
 *
 * @since 1.0
 */
public abstract class UnknownType {

  private String value;

  // Just needed in order to serialize this object
  protected UnknownType() {}

  public UnknownType(String value) {
    this.value = value;
  }

  /**
   * The type is unknown if neither of its possible values is set as true
   *
   * @return {@code true} if the enumerative type used is unknown for this Mule Tooling API Version
   */
  public final boolean isUnknown() {
    return Arrays.stream(this.getClass().getMethods())
        .filter(method -> method.getName().startsWith("is") && !method.getName().equals("isUnknown")).allMatch(method -> {
          try {
            return !((boolean) method.invoke(this));
          } catch (IllegalAccessException | InvocationTargetException e) {
            return false;
          }
        });
  }

  public String getValue() {
    return value;
  }

}
