/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.api.extension.model.config;

import static java.util.Optional.ofNullable;
import org.mule.tooling.client.api.extension.model.DisplayModel;
import org.mule.tooling.client.api.extension.model.ExternalLibraryModel;
import org.mule.tooling.client.api.extension.model.connection.ConnectionProviderModel;
import org.mule.tooling.client.api.extension.model.operation.OperationModel;
import org.mule.tooling.client.api.extension.model.parameter.ParameterGroupModel;
import org.mule.tooling.client.api.extension.model.source.SourceModel;

import java.util.List;
import java.util.Optional;
import java.util.Set;

import org.apache.commons.lang3.builder.ReflectionToStringBuilder;

/**
 * A named configuration for an extension
 * <p/>
 * Configurations describe different ways to initialize a scope for operations.
 * Upon execution, each operation will be associated to a given configuration, so configurations define both
 * a set of shared properties used in operations, and a common context to relate operations.
 * <p/>
 * The configuration can also imply different implicit behaviors not strictly attached to the operations
 * <p/>
 * The configuration is also the place in which cross operation, extension level attributes are configured.
 * Every ExtensionModel is required to have at least one ConfigurationModel.
 * That ConfigurationModel is defined as the &quot;default configuration&quot;
 * <p>
 * Although the SourceModels, OperationModels and ConnectionProviderModels
 * defined at the extension level are available to every single config, configs can also
 * define its own set of those which are exclusive to them.
 *
 * @since 1.0
 */
public class ConfigurationModel {

  private String name;
  private String description;
  private List<ParameterGroupModel> parameterGroupModels;
  private List<OperationModel> operationModels;
  private Set<ExternalLibraryModel> externalLibraryModels;
  private List<SourceModel> sourceModels;
  private List<ConnectionProviderModel> connectionProviders;
  private DisplayModel displayModel;

  // Just needed in order to serialize this object
  private ConfigurationModel() {}

  public ConfigurationModel(String name, String description,
                            List<ParameterGroupModel> parameterGroupModels,
                            List<OperationModel> operationModels,
                            Set<ExternalLibraryModel> externalLibraryModels,
                            List<SourceModel> sourceModels,
                            List<ConnectionProviderModel> connectionProviders,
                            DisplayModel displayModel) {
    this.name = name;
    this.description = description;
    this.parameterGroupModels = parameterGroupModels;
    this.operationModels = operationModels;
    this.externalLibraryModels = externalLibraryModels;
    this.sourceModels = sourceModels;
    this.connectionProviders = connectionProviders;
    this.displayModel = displayModel;
  }

  public String getName() {
    return name;
  }

  public String getDescription() {
    return description;
  }

  public List<ParameterGroupModel> getParameterGroupModels() {
    return parameterGroupModels;
  }

  public List<OperationModel> getOperationModels() {
    return operationModels;
  }

  public Set<ExternalLibraryModel> getExternalLibraryModels() {
    return externalLibraryModels;
  }

  public List<SourceModel> getSourceModels() {
    return sourceModels;
  }

  public List<ConnectionProviderModel> getConnectionProviders() {
    return connectionProviders;
  }

  public Optional<DisplayModel> getDisplayModel() {
    return ofNullable(displayModel);
  }

  @Override
  public int hashCode() {
    return this.name.hashCode();
  }

  @Override
  public boolean equals(Object obj) {
    return this.getClass().isInstance(obj) && this.name.equals(((ConfigurationModel) obj).getName());
  }

  @Override
  public String toString() {
    return ReflectionToStringBuilder.toString(this);
  }

}
