/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.api.metadata.dto;

import java.util.Map;
import java.util.Optional;

import static java.util.Optional.ofNullable;
import static org.mule.runtime.api.util.Preconditions.checkArgument;

public final class MetadataAttributes {

  private final MetadataKey key;
  private final String category;
  private final Map<String, String> parameters;
  private final String outputResolver;
  private final String attributesResolver;

  public MetadataAttributes(MetadataKey key, String category,
                            Map<String, String> parameters, String outputResolver,
                            String attributesResolver) {
    this.key = key;
    this.category = category;
    this.parameters = parameters;
    this.outputResolver = outputResolver;
    this.attributesResolver = attributesResolver;
  }

  /**
   * Provides the {@link MetadataKey} associated (if any) to the resolution of the component's metadata.
   *
   * @return {@link MetadataKey} used for the metadata resolution.
   */
  public Optional<MetadataKey> getKey() {
    return ofNullable(key);
  }

  /**
   * Provides the name of the resolver associated to a given parameter.
   *
   * @param parameterName name of the parameter
   * @return Name of the resolver associated to the parameter
   */
  public String getParameterResolverName(String parameterName) {
    checkArgument(parameters.containsKey(parameterName), "Parameter named %s is not associated to a resolver");
    return parameters.get(parameterName);
  }

  /**
   * Provides the name of the metadata category associated to the Component.
   *
   * @return category name
   */
  public String getCategoryName() {
    return category;
  }

  /**
   * Provides the name of the output resolver (if any) associated to the Component.
   *
   * @return output resolver's name
   */
  public Optional<String> getOutputResolverName() {
    return ofNullable(outputResolver);
  }

  /**
   * Provides the name of the output attributes resolver (if any) associated to the Component.
   *
   * @return output resolver's name
   */
  public Optional<String> getOutputAttributesResolverName() {
    return ofNullable(attributesResolver);
  }
}
