/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.api;

import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.runtime.api.app.declaration.ArtifactDeclaration;
import org.mule.runtime.module.embedded.api.EmbeddedContainer;
import org.mule.tooling.client.api.artifact.ConnectivityTestingServiceBuilder;
import org.mule.tooling.client.api.artifact.ToolingArtifact;
import org.mule.tooling.client.api.artifact.declaration.ArtifactSerializationService;
import org.mule.tooling.client.api.configuration.agent.AgentConfiguration;
import org.mule.tooling.client.api.extension.ExtensionModelService;
import org.mule.tooling.client.api.tryit.TryItService;

import java.net.URL;
import java.util.function.Supplier;

/**
 * Provides tooling services for Mule Runtime.
 *
 * @since 1.0
 */
public interface ToolingRuntimeClient {

  interface Builder {

    /**
     * Sets the {@link AgentConfiguration} used by the client to access the Tooling REST API. If the operations executed by the
     * client don't require to access Tooling REST API this won't need to be set. Otherwise an operation could throw a
     * {@link org.mule.tooling.client.api.exception.MissingToolingConfigurationException} in case if there was no configuration
     * set for the client instance.
     *
     * @param agentConfiguration {@link AgentConfiguration} for the client.
     * @return this
     */
    Builder withRemoteAgentConfiguration(AgentConfiguration agentConfiguration);

    /**
     * Sets the {@link MavenConfiguration} used by the client to access the maven artifacts.
     *
     * If not provided, then the bootstrap maven configuration will be used.
     *
     * @param mavenConfiguration {@link MavenConfiguration} for the client.
     * @return this
     */
    Builder withMavenConfiguration(MavenConfiguration mavenConfiguration);

    /**
     * Running {@link EmbeddedContainer} to use to execute services that requires a runtime.
     * <p>
     * If provided, then it will be used for runtime services. Take into account that the embedded container must have the same
     * mule version and product configuration as this client.
     * <p>
     * It is expected that the container has the agent running and that uses plain HTTP.
     * 
     * @param embeddedContainer an embedded mule container.
     * @param embeddedAgentConfiguration agent configuration for embedded container.
     * @return this
     */
    Builder withEmbeddedContainer(EmbeddedContainer embeddedContainer, AgentConfiguration embeddedAgentConfiguration);

    /**
     * Builds the {@link ToolingRuntimeClient}.
     *
     * @return a new instance of the {@link ToolingRuntimeClient}.
     */
    ToolingRuntimeClient build();

  }

  /**
   * Creates a {@link ToolingArtifact} for the application from the URL of the application content. It can be a file protocol
   * pointing to a the expanded content of the application or a remote location pointing to the zip file where it would be read.
   *
   * @param applicationUrlSupplier {@link Supplier} for the application content URL.
   * @return a {@link ToolingArtifact}.
   */
  ToolingArtifact newToolingArtifact(Supplier<URL> applicationUrlSupplier);

  /**
   * Fetches a {@link ToolingArtifact} for the given applicationId. It also provides the {@link Supplier} for the application
   * content URL. Note: be aware that the application is already deployed on Mule Runtime, it is up to clients to be consistent
   * about the status of the application. There could be inconsistencies if the content of the application supplier differs from
   * the application already deployed on Mule Runtime.
   *
   * @param applicationUrlSupplier {@link Supplier} for the application content URL.
   * @return a {@link ToolingArtifact}.
   */
  ToolingArtifact fetchToolingArtifact(String applicationId, Supplier<URL> applicationUrlSupplier);

  /**
   * Provides a builder to create a dynamically {@link org.mule.tooling.client.api.connectivity.ConnectivityTestingService}.
   *
   * @return a builder to create a {@link ToolingArtifact}
   */
  ConnectivityTestingServiceBuilder newConnectivityTestingService();

  /**
   * Returns a {@link ExtensionModelService} to allow retrieving the {@link org.mule.runtime.api.meta.model.ExtensionModel} of a
   * plugin.
   *
   * @return a {@link ExtensionModelService}
   */
  ExtensionModelService extensionModelService();

  /**
   * Returns an {@link ArtifactSerializationService} that allows to serialize and deserialize an {@link ArtifactDeclaration}.
   *
   * @return a {@link ArtifactSerializationService}
   */
  ArtifactSerializationService artifactSerializationService();

  /**
   * Returns a {@link TryItService} to do try it over an application.
   *
   * @return a {@link TryItService}
   */
  TryItService tryItService();
}
