/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.bootstrap;

import static com.google.common.io.Files.getFileExtension;
import static java.lang.Thread.currentThread;
import static java.util.Objects.requireNonNull;
import static java.util.ServiceLoader.load;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.ArrayUtils.addAll;
import static org.mule.tooling.client.api.descriptors.ArtifactDescriptor.newBuilder;
import static org.mule.tooling.client.bootstrap.internal.VersionResolver.mapToolingVersion;
import org.mule.maven.client.api.MavenClient;
import org.mule.maven.client.api.MavenClientProvider;
import org.mule.maven.client.api.model.BundleDependency;
import org.mule.maven.client.api.model.BundleDescriptor;
import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.tooling.client.api.AbstractToolingRuntimeClientBuilderFactory;
import org.mule.tooling.client.api.Disposable;
import org.mule.tooling.client.api.ToolingRuntimeClient;
import org.mule.tooling.client.api.ToolingRuntimeClient.Builder;
import org.mule.tooling.client.api.descriptors.ArtifactDescriptor;
import org.mule.tooling.client.api.exception.ToolingException;

import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.ServiceLoader;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Bootstrap for {@link ToolingRuntimeClient}.
 *
 * @since 1.0
 */
public class ToolingRuntimeClientBootstrap implements Disposable {

  private static final String ORG_MULE_TOOLING = "org.mule.tooling";
  private static final String MULE_RUNTIME_TOOLING_CLIENT = "mule-runtime-tooling-client";

  private final Logger logger = LoggerFactory.getLogger(this.getClass());
  private final MavenClient mavenClient;

  private URLClassLoader extendedClassLoader;
  private AbstractToolingRuntimeClientBuilderFactory toolingRuntimeClientBuilderFactory;

  /**
   * Creates an instance of the datasense.
   *
   * @param muleVersion defines the Mule Runtime version which the client should be created for.
   * @param mavenConfiguration maven configuration to be used for locating the runtime artifacts. This configuration is not to
   *        load extension model
   */
  public ToolingRuntimeClientBootstrap(String muleVersion, MavenConfiguration mavenConfiguration) {
    requireNonNull(muleVersion, "muleVersion cannot be null");
    logger.info("Creating loader for muleVersion: {}", muleVersion);

    mavenClient = MavenClientProvider.discoverProvider(ToolingRuntimeClientBootstrap.class.getClassLoader())
        .createMavenClient(mavenConfiguration);
    this.extendedClassLoader = createClassLoader(muleVersion);
    this.toolingRuntimeClientBuilderFactory = discoverToolingRuntimeClientFactory();
  }

  private AbstractToolingRuntimeClientBuilderFactory discoverToolingRuntimeClientFactory() {
    ServiceLoader<AbstractToolingRuntimeClientBuilderFactory> serviceLoader =
        load(AbstractToolingRuntimeClientBuilderFactory.class, extendedClassLoader);
    if (!serviceLoader.iterator().hasNext()) {
      throw new IllegalStateException("No service found for: '" + AbstractToolingRuntimeClientBuilderFactory.class.getName()
          + "'");
    }

    return serviceLoader.iterator().next();
  }

  private URLClassLoader createClassLoader(String muleVersion) {
    try {
      if (logger.isDebugEnabled()) {
        logger.debug("Creating URL class loader for muleVersion: {}", muleVersion);
      }

      String toolingVersion = mapToolingVersion(this.getClass().getClassLoader(), muleVersion);

      Set<File> classpath = new LinkedHashSet<>();
      classpath.addAll(resolveDependency(newBuilder()
          .withGroupId(ORG_MULE_TOOLING)
          .withArtifactId(MULE_RUNTIME_TOOLING_CLIENT)
          .withVersion(toolingVersion)
          .build()));

      BundleDependency bundleDependency =
          mavenClient.resolveBundleDescriptor(new BundleDescriptor.Builder().setGroupId(ORG_MULE_TOOLING)
              .setArtifactId(MULE_RUNTIME_TOOLING_CLIENT)
              .setVersion(toolingVersion).setType("jar").build());

      URL[] urls = classpath.stream()
          .filter(file -> getFileExtension(file.getAbsolutePath()).equals("jar"))
          .map(file -> toUrl(file))
          .toArray(URL[]::new);
      urls = addAll(urls, getBundleUrl(bundleDependency));
      URLClassLoader classLoader = new ExtendedToolingURLClassLoader(urls, this.getClass().getClassLoader());

      if (logger.isTraceEnabled()) {
        logger.trace("Created extended URL class loader: {}", classpath);
      }

      return classLoader;
    } catch (Exception e) {
      throw new ToolingException("Couldn't create the class loader for the tooling client", e);
    }
  }

  private URL getBundleUrl(BundleDependency bundleDependency) {
    try {
      return bundleDependency.getBundleUri().toURL();
    } catch (MalformedURLException e) {
      throw new MuleRuntimeException(e);
    }
  }

  private List<File> resolveDependency(ArtifactDescriptor artifactDescriptor) {
    if (logger.isDebugEnabled()) {
      logger.debug("Resolving Tooling Runtime Client implementation libraries from: {}", artifactDescriptor);
    }

    List<BundleDependency> bundleDependencies = mavenClient
        .resolveBundleDescriptorDependencies(false, true,
                                             new BundleDescriptor.Builder()
                                                 .setGroupId(artifactDescriptor.getGroupId())
                                                 .setArtifactId(artifactDescriptor.getArtifactId())
                                                 .setVersion(artifactDescriptor.getVersion())
                                                 .setType(artifactDescriptor.getExtension())
                                                 .setClassifier(artifactDescriptor.getClassifier())
                                                 .build());

    return bundleDependencies.stream()
        .filter(bundleDependency -> bundleDependency.getBundleUri() != null)
        .map(bundleDependency -> new File(getBundleUrl(bundleDependency).getFile()))
        .collect(toList());
  }

  private static URL toUrl(File file) {
    try {
      return file.toURI().toURL();
    } catch (MalformedURLException e) {
      throw new ToolingException("Couldn't get URL for file: " + file, e);
    }
  }

  /**
   * Creates a {@link Builder} by creating an extended {@link ClassLoader} with the Mule Runtime dependencies.
   *
   * @return a {@link Builder}
   */
  public Builder newToolingRuntimeClientBuilder() {
    if (extendedClassLoader == null) {
      throw new IllegalStateException("Cannot be created a ToolingRuntimeClient builder once the bootstrap has been disposed");
    }
    ClassLoader currentContextClassLoader = currentThread().getContextClassLoader();
    try {
      currentThread().setContextClassLoader(extendedClassLoader);
      return new ExtendedBuilder(toolingRuntimeClientBuilderFactory.create(mavenClient), extendedClassLoader);
    } catch (Exception e) {
      throw new ToolingException("Error while instantiating DefaultToolingRuntimeClient class", e);
    } finally {
      currentThread().setContextClassLoader(currentContextClassLoader);
    }
  }

  @Override
  public void dispose() {
    try {
      toolingRuntimeClientBuilderFactory.dispose();
    } finally {
      toolingRuntimeClientBuilderFactory = null;
    }
    try {
      extendedClassLoader.close();
    } catch (IOException e) {
      throw new ToolingException("Error while closing extended class loader", e);
    } finally {
      extendedClassLoader = null;
    }
  }

}
