/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.bootstrap;

import java.io.IOException;
import java.io.InputStream;
import java.net.JarURLConnection;
import java.net.URL;
import java.net.URLClassLoader;
import java.net.URLStreamHandler;
import java.net.URLStreamHandlerFactory;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import sun.net.www.protocol.jar.Handler;

/**
 * Child first implementation for the extended Tooling class loader.
 *
 * @since 1.0
 */
public class ExtendedToolingURLClassLoader extends URLClassLoader {

  private final Logger logger = LoggerFactory.getLogger(this.getClass());

  private ClassLoader system;

  /**
   * Creates an extended child first {@link ClassLoader}.
   *
   * @param classpath {@link List} of {@link URL}s to be loaded by this class loader.
   * @param parent the {@link ClassLoader} parent.
   */
  public ExtendedToolingURLClassLoader(URL[] classpath, ClassLoader parent) {
    super(classpath, parent, new NonCachingURLStreamHandlerFactory());
    system = getSystemClassLoader();
    if (logger.isTraceEnabled()) {
      logger.trace("SystemClassLoader is {}", system);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected synchronized Class<?> loadClass(String name, boolean resolve)
      throws ClassNotFoundException {
    Class<?> clazz = findLoadedClass(name);
    if (clazz == null) {
      if (system != null) {
        try {
          clazz = system.loadClass(name);
          if (logger.isTraceEnabled()) {
            logClassLoaded(name, system);
          }
        } catch (ClassNotFoundException ignored) {
        }
      }
      if (clazz == null) {
        try {
          clazz = findClass(name);
          if (logger.isTraceEnabled()) {
            logClassLoaded(name, this);
          }
        } catch (ClassNotFoundException e) {
          clazz = super.loadClass(name, resolve);
          if (logger.isTraceEnabled()) {
            logClassLoaded(name, this.getParent());
          }
        }
      }
    }
    if (resolve) {
      resolveClass(clazz);
    }
    return clazz;
  }

  private void logClassLoaded(String className, ClassLoader classLoader) {
    logger.trace("Class '{}' loaded with {}", className, classLoader);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public URL getResource(String name) {
    URL url = null;
    if (system != null) {
      url = system.getResource(name);
    }
    if (url == null) {
      url = findResource(name);
      if (url == null) {
        url = super.getResource(name);
      }
    }
    return url;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Enumeration<URL> getResources(String name) throws IOException {
    Enumeration<URL> systemUrls = null;
    if (system != null) {
      systemUrls = system.getResources(name);
    }
    Enumeration<URL> localUrls = findResources(name);
    Enumeration<URL> parentUrls = null;
    if (getParent() != null && getParent() != system) {
      parentUrls = getParent().getResources(name);
    }
    final List<URL> urls = new ArrayList<>();
    if (systemUrls != null) {
      while (systemUrls.hasMoreElements()) {
        urls.add(systemUrls.nextElement());
      }
    }
    if (localUrls != null) {
      while (localUrls.hasMoreElements()) {
        urls.add(localUrls.nextElement());
      }
    }
    if (parentUrls != null) {
      while (parentUrls.hasMoreElements()) {
        urls.add(parentUrls.nextElement());
      }
    }
    return new Enumeration<URL>() {

      Iterator<URL> iter = urls.iterator();

      public boolean hasMoreElements() {
        return iter.hasNext();
      }

      public URL nextElement() {
        return iter.next();
      }

    };
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public InputStream getResourceAsStream(String name) {
    URL url = getResource(name);
    try {
      return url != null ? url.openStream() : null;
    } catch (IOException e) {
    }
    return null;
  }

  protected static class NonCachingURLStreamHandlerFactory implements URLStreamHandlerFactory {

    @Override
    public URLStreamHandler createURLStreamHandler(String protocol) {
      return new NonCachingJarResourceURLStreamHandler();
    }
  }

  /**
   * Prevents jar caching for this classloader, mainly to fix the static ResourceBundle mess/cache that keeps connections open no
   * matter what.
   */
  private static class NonCachingJarResourceURLStreamHandler extends Handler {

    public NonCachingJarResourceURLStreamHandler() {
      super();
    }

    @Override
    protected java.net.URLConnection openConnection(URL u) throws IOException {
      JarURLConnection c = new sun.net.www.protocol.jar.JarURLConnection(u, this);
      c.setUseCaches(false);
      return c;
    }
  }


}
