/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.bootstrap;

import static java.lang.Thread.currentThread;
import static java.util.Objects.requireNonNull;
import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.tooling.client.api.ToolingRuntimeClient;
import org.mule.tooling.client.api.ToolingRuntimeClient.Builder;
import org.mule.tooling.client.api.configuration.agent.AgentConfiguration;

import java.io.InputStream;
import java.net.URLClassLoader;

/**
 * Extends the default builder in order to close the {@link ClassLoader extendedClassLoader} when {@link ToolingRuntimeClient} is
 * disposed.
 *
 * @since 1.0
 */
class ExtendedBuilder implements Builder {

  private Builder delegate;
  private URLClassLoader extendedClassLoader;

  /**
   * Creates an instace of the builder.
   *
   * @param builder {@link Builder} to delegate the creating of the client.
   * @param extendedClassLoader {@link URLClassLoader} with Mule Runtime libraries.
   */
  ExtendedBuilder(Builder builder, URLClassLoader extendedClassLoader) {
    requireNonNull(builder, "builder cannot be null");
    requireNonNull(extendedClassLoader, "extendedClassLoader cannot be null");

    if (builder.getClass().getClassLoader() != extendedClassLoader) {
      throw new IllegalStateException("Builder should be created with the same extended class loader");
    }

    this.delegate = builder;
    this.extendedClassLoader = extendedClassLoader;
  }

  @Override
  public Builder withRemoteAgentConfiguration(AgentConfiguration agentConfiguration) {
    this.delegate.withRemoteAgentConfiguration(agentConfiguration);
    return this;
  }

  @Override
  public Builder withMavenConfiguration(MavenConfiguration mavenConfiguration) {
    this.delegate.withMavenConfiguration(mavenConfiguration);
    return this;
  }

  @Override
  public Builder withLoggingConfiguration(InputStream input) {
    this.delegate.withLoggingConfiguration(input);
    return this;
  }

  @Override
  public ToolingRuntimeClient build() {
    ClassLoader currentContextClassLoader = currentThread().getContextClassLoader();
    try {
      currentThread().setContextClassLoader(extendedClassLoader);
      return this.delegate.build();
    } finally {
      currentThread().setContextClassLoader(currentContextClassLoader);
    }
  }
}
