/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Throwables.propagateIfPossible;
import static java.lang.String.format;
import static java.util.Optional.ofNullable;
import static org.apache.commons.io.FileUtils.toFile;
import static org.mule.runtime.core.api.util.ClassUtils.withContextClassLoader;
import static org.mule.tooling.client.internal.DataSensePartsFactory.toDataSenseComponentInfoDTO;
import static org.mule.tooling.client.internal.DataSensePartsFactory.toDataSenseInfoDTO;
import org.mule.datasense.api.DataSense;
import org.mule.datasense.api.DataSenseInfo;
import org.mule.datasense.api.DataSenseResolutionScope;
import org.mule.datasense.api.metadataprovider.ApplicationModel;
import org.mule.datasense.api.metadataprovider.CachedDataSenseProvider;
import org.mule.datasense.api.metadataprovider.DataSenseMetadataCacheProvider;
import org.mule.datasense.api.metadataprovider.DataSenseProvider;
import org.mule.datasense.api.metadataprovider.DefaultApplicationModel;
import org.mule.datasense.api.metadataprovider.DefaultDataSenseProvider;
import org.mule.datasense.api.metadataprovider.ExtensionsProvider;
import org.mule.datasense.impl.DefaultDataSense;
import org.mule.runtime.api.component.location.Location;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.tooling.client.api.datasense.DataSenseComponentInfo;
import org.mule.tooling.client.api.datasense.DataSenseRequest;
import org.mule.tooling.client.api.datasense.DataSenseResolveRequest;
import org.mule.tooling.client.api.datasense.DataSenseService;
import org.mule.tooling.client.api.exception.MissingToolingConfigurationException;
import org.mule.tooling.client.api.exception.ToolingException;
import org.mule.tooling.client.internal.application.Application;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableSet;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicBoolean;

import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Default implementation of {@link DataSenseService}.
 *
 * @since 4.0
 */
public class DefaultDataSenseService implements DataSenseService {

  private final Logger logger = LoggerFactory.getLogger(this.getClass());

  private static final String APP_TYPES_DATA = "application-types.xml";

  private DataSense dataSense = new DefaultDataSense();

  private Application application;
  private MetadataProvider metadataProvider;

  private AtomicBoolean disposed = new AtomicBoolean(false);

  /**
   * Creates an instance of the DataSense service.
   *
   * @param application {@link Application} to resolve DataSense. Non null.
   * @param metadataProvider {@link MetadataProvider} to resolve dynamic Metadata if needed. Non null.
   */
  public DefaultDataSenseService(Application application, MetadataProvider metadataProvider) {
    checkNotNull(application, "application cannot be null");
    checkNotNull(metadataProvider, "metadataProvider cannot be null");

    this.application = application;
    this.metadataProvider = metadataProvider;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Optional<org.mule.tooling.client.api.datasense.DataSenseInfo> resolveDataSense(DataSenseRequest dataSenseRequest) {
    checkState();
    return toDataSenseInfoDTO(internalResolveDataSense(dataSenseRequest));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Optional<DataSenseComponentInfo> resolveComponentDataSense(DataSenseRequest dataSenseRequest) {
    checkState();
    return toDataSenseComponentInfoDTO(internalResolveComponentDataSense(dataSenseRequest));
  }

  @Override
  public Optional<DataSenseInfo> internalResolveDataSense(DataSenseRequest dataSenseRequest) {
    checkState();
    Location location = dataSenseRequest.getLocation();

    DataSenseMetadataCacheProvider dataSenseMetadataCacheProvider = dataSenseRequest.getDataSenseMetadataCacheProvider();

    if (logger.isDebugEnabled()) {
      logger.debug(format("Resolving DataSense for component location: %s on %s", dataSenseRequest.getLocation(), application));
    }

    try {
      ApplicationModel applicationModel = buildDataSenseApplicationModel(application);

      DataSenseProvider dataSenseProvider =
          getDataSenseProvider(applicationModel,
                               () -> ImmutableSet.<ExtensionModel>builder().addAll(application.getExtensionModels()).build(),
                               dataSenseMetadataCacheProvider,
                               dataSenseRequest);

      DataSenseResolutionScope dataSenseResolutionScope = dataSenseRequest instanceof DataSenseResolveRequest
          ? ((DataSenseResolveRequest) dataSenseRequest).getDataSenseResolutionScope() : null;
      if (dataSenseResolutionScope == null) {
        return withContextClassLoader(application.getArtifactClassLoader().getClassLoader(),
                                      () -> this.dataSense.resolve(location, applicationModel, dataSenseProvider));
      } else {
        return withContextClassLoader(application.getArtifactClassLoader().getClassLoader(),
                                      () -> this.dataSense.resolve(dataSenseResolutionScope, applicationModel,
                                                                   dataSenseProvider));
      }
    } catch (Exception e) {
      propagateIfPossible(e, MissingToolingConfigurationException.class);
      throw new ToolingException(format("Error while resolving DataSense for location: %s on %s", location, application), e);
    }
  }

  @Override
  public Optional<org.mule.datasense.api.DataSenseComponentInfo> internalResolveComponentDataSense(DataSenseRequest dataSenseRequest) {
    checkState();
    Location location = dataSenseRequest.getLocation();
    checkNotNull(location, "location cannot be null");

    DataSenseMetadataCacheProvider dataSenseMetadataCacheProvider = dataSenseRequest.getDataSenseMetadataCacheProvider();

    if (logger.isDebugEnabled()) {
      logger.debug(format("Resolving DataSense for location: %s on %s", dataSenseRequest.getLocation(), application));
    }

    try {
      ApplicationModel applicationModel = buildDataSenseApplicationModel(application);
      DataSenseProvider dataSenseProvider =
          getDataSenseProvider(applicationModel,
                               () -> ImmutableSet.<ExtensionModel>builder().addAll(application.getExtensionModels()).build(),
                               dataSenseMetadataCacheProvider,
                               dataSenseRequest);

      return withContextClassLoader(application.getArtifactClassLoader().getClassLoader(),
                                    () -> this.dataSense.resolveComponent(location, applicationModel, dataSenseProvider));
    } catch (Exception e) {
      propagateIfPossible(e, MissingToolingConfigurationException.class);
      throw new ToolingException(format("Error while resolving DataSense for location %s on %s", location, application), e);
    }
  }

  public void dispose() {
    disposed.compareAndSet(false, true);
  }

  private ApplicationModel buildDataSenseApplicationModel(Application application) {
    String applicationName = application.getApplicationName();
    org.mule.runtime.config.internal.model.ApplicationModel applicationModel = application.getApplicationModel();

    List<String> appTypesList = new ArrayList<>();
    InputStream appTypeData = application.getArtifactClassLoader().getClassLoader().getResourceAsStream(APP_TYPES_DATA);
    if (appTypeData != null) {
      logger.debug("Found application custom types data");
      try {
        appTypesList.add(IOUtils.toString(appTypeData));
      } catch (IOException e) {
        throw new ToolingException(format("Error while reading application custom types file for %s", application), e);
      }
    }
    return new DefaultApplicationModel(applicationName,
                                       applicationModel,
                                       appTypesList,
                                       resource -> ofNullable(
                                                              application.getArtifactClassLoader().getClassLoader()
                                                                  .getResource(resource)).map(url -> toFile(url).toURI()));
  }

  private DataSenseProvider getDataSenseProvider(ApplicationModel applicationModel, ExtensionsProvider extensionsProvider,
                                                 DataSenseMetadataCacheProvider dataSenseMetadataCacheProvider,
                                                 DataSenseRequest request) {
    DataSenseProvider dataSenseProvider =
        new DefaultDataSenseProvider(extensionsProvider,
                                     new ToolingDataSenseMetadataProvider(metadataProvider,
                                                                          request.getRequestTimeout()));
    if (dataSenseMetadataCacheProvider != null) {
      dataSenseProvider = new CachedDataSenseProvider(applicationModel, dataSenseMetadataCacheProvider, dataSenseProvider);
    }
    return dataSenseProvider;
  }

  private void checkState() {
    Preconditions.checkState(!disposed.get(), "Service already disposed, cannot be used anymore");
  }

}
