/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static java.util.Collections.sort;
import static java.util.Comparator.comparingLong;
import static java.util.function.Function.identity;
import static java.util.Objects.requireNonNull;
import static java.util.stream.Collectors.toList;
import org.mule.runtime.api.util.LazyValue;
import org.mule.tooling.agent.RuntimeToolingService;
import org.mule.tooling.agent.rest.client.tooling.applications.applicationName.tryIt.AgentTrackingNotificationResponse;
import org.mule.tooling.client.api.tryit.MessageHistory;
import org.mule.tooling.client.api.tryit.TryItService;
import static java.util.stream.Collectors.toMap;
import org.mule.tooling.client.api.types.Transaction;
import org.mule.tooling.client.api.types.TransactionStackEntry;
import org.mule.tooling.client.api.types.TransactionStatus;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.mule.tooling.event.model.component.location.ComponentLocation;

/**
 * Default implementation for {@link org.mule.tooling.client.api.tryit.TryItService}.
 *
 * @since 4.0
 */
public class DefaultTryItService implements TryItService {

  private final Logger logger = LoggerFactory.getLogger(this.getClass());

  private static final String FIRST_MESSAGE_PROCESSOR_INDEX = "/0";
  private static final String MESSAGE_PRE_INVOKE_ACTION = "message processor pre invoke";
  private static final String MESSAGE_POST_INVOKE_ACTION = "message processor post invoke";
  private static final String MESSAGE_EXCEPTION = "exception";

  private LazyValue<RuntimeToolingService> runtimeToolingServiceLazyValue;

  /**
   * Creates a default instance of the service.
   *
   * @param runtimeToolingServiceLazyValue {@link LazyValue} for {@link RuntimeToolingService}.
   */
  protected DefaultTryItService(LazyValue<RuntimeToolingService> runtimeToolingServiceLazyValue) {
    requireNonNull(runtimeToolingServiceLazyValue, "runtimeToolingServiceLazyValue cannot be null");

    this.runtimeToolingServiceLazyValue = runtimeToolingServiceLazyValue;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void enable(String applicationName) {
    runtimeToolingServiceLazyValue.get().enableTryIt(applicationName);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public MessageHistory consume(String applicationName, int chunkSize) {
    List<AgentTrackingNotificationResponse> agentTrackingNotifications =
        runtimeToolingServiceLazyValue.get().consumeTryItNotifications(applicationName, chunkSize);
    return convertToMessageHistory(agentTrackingNotifications);
  }

  private MessageHistory convertToMessageHistory(List<AgentTrackingNotificationResponse> agentTrackingNotifications) {
    MessageHistory messageHistory = new MessageHistory();
    List<Transaction> transactions = new ArrayList<>();
    messageHistory.setTransactions(transactions);
    Map<String, List<AgentTrackingNotificationResponse>> notificationsMap =
        splitAgentNotificationsAndSortByDate(agentTrackingNotifications);
    notificationsMap.keySet().stream().forEach(transactionId -> {
      List<AgentTrackingNotificationResponse> transactionNotifications = notificationsMap.get(transactionId);
      Transaction transaction = new Transaction();
      // TODO MULE-12229 Support TransactionStatus
      transaction.setTransactionStatus(TransactionStatus.COMPLETE);
      transaction.setId(transactionNotifications.get(0).getTransactionId());
      AgentTrackingNotificationResponse firstMessageProcessorInputNotification = transactionNotifications.get(0);
      if (firstMessageProcessorInputNotification.getComponentLocation().getLocation().endsWith(FIRST_MESSAGE_PROCESSOR_INDEX) &&
          firstMessageProcessorInputNotification.getAction().equals(MESSAGE_PRE_INVOKE_ACTION)) {
        transaction.setMessage(firstMessageProcessorInputNotification.getEvent().getMessage());
        transaction.setTimestamp(firstMessageProcessorInputNotification.getTimestamp());
        transaction.setGlobalName(getGlobalNameFromLocation(firstMessageProcessorInputNotification.getComponentLocation()
            .getLocation()));
      }

      markAsUnsuccessfulExceptionResponses(transactionNotifications);
      markAsUnsuccesfulUnfinishedResponses(transactionNotifications);

      transaction.setTransactionStack(transactionNotifications.stream()
          .filter(agentTrackingNotification -> agentTrackingNotification.getAction().equals(MESSAGE_POST_INVOKE_ACTION))
          .map(agentTrackingNotification -> {
            TransactionStackEntry transactionStackEntry = new TransactionStackEntry();
            transactionStackEntry.setTimestamp(agentTrackingNotification.getTimestamp());
            transactionStackEntry.setComponentLocation(agentTrackingNotification.getComponentLocation());
            transactionStackEntry.setEventModel(agentTrackingNotification.getEvent());
            return transactionStackEntry;
          }).collect(toList()));
      transactions.add(transaction);
    });
    return messageHistory;
  }

  private Map<String, List<AgentTrackingNotificationResponse>> splitAgentNotificationsAndSortByDate(List<AgentTrackingNotificationResponse> agentTrackingNotifications) {
    logger.debug("Grouping notifications by correlationId");
    Map<String, List<AgentTrackingNotificationResponse>> notificationsMap = new HashMap<>();
    agentTrackingNotifications.stream().forEach(agentTrackingNotification -> {
      logger.debug("Processing notification: {}", agentTrackingNotification);
      List<AgentTrackingNotificationResponse> notifications = notificationsMap.get(agentTrackingNotification.getCorrelationId());
      if (notifications == null) {
        notifications = new ArrayList<>();
        notificationsMap.put(agentTrackingNotification.getCorrelationId(), notifications);
      }
      notifications.add(agentTrackingNotification);
    });

    for (List<AgentTrackingNotificationResponse> transactinAgentNotifications : notificationsMap.values()) {
      sort(transactinAgentNotifications, comparingLong(AgentTrackingNotificationResponse::getTimestamp));
    }
    return notificationsMap;
  }

  private static Map<String, AgentTrackingNotificationResponse> getPostInvokeNotifications(List<AgentTrackingNotificationResponse> agentTrackingNotificationResponses) {
    return agentTrackingNotificationResponses.stream()
        .filter(notification -> notification.getAction().equals(MESSAGE_POST_INVOKE_ACTION))
        .collect(toMap(notification -> notification.getComponentLocation().getLocation(), identity()));
  }

  private static Map<String, AgentTrackingNotificationResponse> getExceptionNotifications(List<AgentTrackingNotificationResponse> agentTrackingNotificationResponses) {
    return agentTrackingNotificationResponses.stream().filter(notification -> notification.getAction().equals(MESSAGE_EXCEPTION))
        .collect(toMap(AgentTrackingNotificationResponse::getTransactionId, identity()));
  }

  private static void markAsUnsuccessfulExceptionResponses(List<AgentTrackingNotificationResponse> agentTrackingNotificationResponses) {
    List<AgentTrackingNotificationResponse> createdExceptionResponses = new ArrayList<>();

    Map<String, AgentTrackingNotificationResponse> postInvokeNotifications =
        getPostInvokeNotifications(agentTrackingNotificationResponses);

    agentTrackingNotificationResponses.stream()
        .filter(agentTrackingNotificationResponse -> agentTrackingNotificationResponse.getAction().equals(MESSAGE_EXCEPTION))
        .forEach(exceptionNotificationResponse -> {
          ComponentLocation componentLocation = exceptionNotificationResponse.getComponentLocation();

          // In case the notifications where polled before one or more POST INVOKE notifications where created,
          // then we need to have a way to show that the notification is unsuccessful until the Completion Event
          // is handled
          if (postInvokeNotifications.containsKey(componentLocation.getLocation())) {
            postInvokeNotifications.get(componentLocation.getLocation()).getEvent().setSuccessful(false);
          } else {
            createdExceptionResponses.add(createResponseFromExceptionEvent(exceptionNotificationResponse));
          }
        });
    agentTrackingNotificationResponses.addAll(createdExceptionResponses);
  }

  private static void markAsUnsuccesfulUnfinishedResponses(List<AgentTrackingNotificationResponse> agentTrackingNotificationResponses) {
    List<AgentTrackingNotificationResponse> createdUnfinishedResponses = new ArrayList<>();

    Map<String, AgentTrackingNotificationResponse> postInvokeNotifications =
        getPostInvokeNotifications(agentTrackingNotificationResponses);
    Map<String, AgentTrackingNotificationResponse> exceptionNotifications =
        getExceptionNotifications(agentTrackingNotificationResponses);

    agentTrackingNotificationResponses.stream().filter(agentTrackingNotificationResponse -> agentTrackingNotificationResponse
        .getAction().equals(MESSAGE_PRE_INVOKE_ACTION))
        .forEach(agentTrackingNotificationResponse -> {
          // If there is a post-invoke notification associated to this pre-invoke, there is nothing to do
          if (postInvokeNotifications.containsKey(agentTrackingNotificationResponse.getComponentLocation().getLocation())) {
            return;
          }

          // If there isn't a post-invoke, but there is an exception notification that doesn't match the location
          // (as taken into account in the other scenario), then this notification is unfinished (e.g. execution of
          // smart-connector). Then, exception notification's transaction ID will match this pre-invoke's transaction ID
          String transactionID = agentTrackingNotificationResponse.getTransactionId();
          if (exceptionNotifications.containsKey(transactionID)) {
            AgentTrackingNotificationResponse errorResponse =
                createResponseFromExceptionEvent(exceptionNotifications.get(transactionID));
            errorResponse.setComponentLocation(agentTrackingNotificationResponse.getComponentLocation());
            createdUnfinishedResponses.add(errorResponse);
          }
        });
    agentTrackingNotificationResponses.addAll(createdUnfinishedResponses);
  }

  private static AgentTrackingNotificationResponse createResponseFromExceptionEvent(AgentTrackingNotificationResponse exceptionNotificationResponse) {
    AgentTrackingNotificationResponse response = new AgentTrackingNotificationResponse();
    response.setAction(MESSAGE_POST_INVOKE_ACTION);
    response.setComponentLocation(exceptionNotificationResponse.getComponentLocation());
    response.setEvent(exceptionNotificationResponse.getEvent());
    response.setTimestamp(exceptionNotificationResponse.getTimestamp());

    return response;
  }

  private static String getGlobalNameFromLocation(String location) {
    return org.mule.tooling.client.api.location.Location.builderFromStringRepresentation(location).build().getGlobalName();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void disable(String applicationName) {
    try {
      runtimeToolingServiceLazyValue.get().disableTryIt(applicationName);
    } catch (Exception e) {
      logger.warn("Error while disabling application for try it", e);
    }
  }

}
