/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal;

import static com.google.common.base.Preconditions.checkNotNull;
import static org.mule.runtime.api.value.ResolvingFailure.Builder.newFailure;
import static org.mule.runtime.api.value.ValueResult.resultFrom;
import static org.mule.runtime.extension.api.values.ValueResolvingException.UNKNOWN;
import org.mule.runtime.api.value.ResolvingFailure;
import org.mule.runtime.api.value.Value;
import org.mule.runtime.api.value.ValueResult;
import org.mule.runtime.deployment.model.api.DeploymentException;
import org.mule.tooling.client.api.value.provider.ValueProviderRequest;
import org.mule.tooling.client.api.value.provider.ValueProviderService;
import org.mule.tooling.client.internal.application.Application;
import org.mule.tooling.client.internal.application.RemoteApplicationInvoker;

import java.util.concurrent.CompletionException;

/**
 * Provides the capability of resolve the {@link Value values} for any Value Provider capable element in a Mule App.
 *
 * @since 1.0
 */
public class DefaultValueProviderService implements ValueProviderService {

  private static final String DEPLOYMENT = "DEPLOYMENT";
  private RemoteApplicationInvoker remoteApplicationInvoker;

  /**
   * Creates an instance of the service.
   *
   * @param remoteApplicationInvoker {@link Application} to deploy and resolve operations against a Mule Runtime. Non null.
   */
  public DefaultValueProviderService(RemoteApplicationInvoker remoteApplicationInvoker) {
    checkNotNull(remoteApplicationInvoker, "remoteApplicationInvoker cannot be null");

    this.remoteApplicationInvoker = remoteApplicationInvoker;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ValueResult getValues(ValueProviderRequest valueProviderRequest) {
    try {
      return remoteApplicationInvoker.evaluateWithRemoteApplication((applicationId,
                                                                     runtimeToolingService) -> runtimeToolingService
                                                                         .getValues(applicationId,
                                                                                    valueProviderRequest
                                                                                        .getLocation(),
                                                                                    valueProviderRequest
                                                                                        .getProviderName()));
    } catch (Throwable t) {
      return resultFrom(getFailure(t));
    }
  }

  ResolvingFailure getFailure(Throwable throwable) {
    String failureCode = UNKNOWN;
    if (throwable instanceof CompletionException) {
      if (throwable.getCause() instanceof DeploymentException) {
        failureCode = DEPLOYMENT;
      }
    }
    return newFailure(throwable).withFailureCode(failureCode).build();
  }
}
