/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static com.google.common.collect.Maps.newHashMap;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;
import static org.mule.tooling.client.internal.CollectionUtils.copy;
import static org.mule.tooling.client.internal.ExtensionModelPartsFactory.toConnectionProviderDTO;
import static org.mule.tooling.client.internal.ExtensionModelPartsFactory.toConstructModelDTO;
import static org.mule.tooling.client.internal.ExtensionModelPartsFactory.toDisplayModelTO;
import static org.mule.tooling.client.internal.ExtensionModelPartsFactory.toExternalLibraryModelsDTO;
import static org.mule.tooling.client.internal.ExtensionModelPartsFactory.toFunctionModelDTO;
import static org.mule.tooling.client.internal.ExtensionModelPartsFactory.toOperationModelDTO;
import static org.mule.tooling.client.internal.ExtensionModelPartsFactory.toParameterGroupModelsDTO;
import static org.mule.tooling.client.internal.ExtensionModelPartsFactory.toSourceModelDTO;
import static org.mule.tooling.client.internal.ExtensionModelPartsFactory.toStereotypeDTO;

import org.mule.tooling.client.api.extension.model.Category;
import org.mule.tooling.client.api.extension.model.ErrorModel;
import org.mule.tooling.client.api.extension.model.ExtensionModel;
import org.mule.tooling.client.api.extension.model.ImportedTypeModel;
import org.mule.tooling.client.api.extension.model.SubTypesModel;
import org.mule.tooling.client.api.extension.model.XmlDslModel;
import org.mule.tooling.client.api.extension.model.config.ConfigurationModel;
import org.mule.tooling.client.api.extension.model.connection.ConnectionProviderModel;
import org.mule.tooling.client.api.extension.model.construct.ConstructModel;
import org.mule.tooling.client.api.extension.model.function.FunctionModel;
import org.mule.tooling.client.api.extension.model.operation.OperationModel;
import org.mule.tooling.client.api.extension.model.source.SourceModel;

import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Factory for {@link org.mule.tooling.client.api.extension.model.ExtensionModel}. Responsible for adapting a Mule Runtime
 * {@link org.mule.runtime.api.meta.model.ExtensionModel} and implements flyweight pattern in order to reduce the amount of
 * objects created.
 *
 * @since 4.0
 */
public class ExtensionModelFactory {

  // Caches for implementing flyweight pattern
  private Map<String, ConnectionProviderModel> connectionProviderModelsCache = newHashMap();
  private Map<String, SourceModel> sourceModelsCache = newHashMap();
  private Map<String, OperationModel> operationModelsCache = newHashMap();
  private Map<String, ConstructModel> constructModelsCache = newHashMap();
  private Map<String, FunctionModel> functionModelsCache = newHashMap();
  private Map<String, ErrorModel> errorModelsCache = newHashMap();


  public ExtensionModel createExtensionModel(org.mule.runtime.api.meta.model.ExtensionModel runtimeExtensionModel,
                                             String minMuleVersion) {
    return new ExtensionModel(
                              runtimeExtensionModel.getName(),
                              toCategoryDTO(runtimeExtensionModel.getCategory()),
                              runtimeExtensionModel.getVendor(),
                              copy(runtimeExtensionModel.getResources()),
                              runtimeExtensionModel.getImportedTypes().stream()
                                  .map(importedTypeModel -> new ImportedTypeModel(importedTypeModel.getImportedType())).collect(
                                                                                                                                toSet()),
                              runtimeExtensionModel
                                  .getSubTypes().stream().map(subTypesModel -> new SubTypesModel(subTypesModel.getBaseType(),
                                                                                                 subTypesModel.getSubTypes()))
                                  .collect(
                                           toSet()),
                              runtimeExtensionModel.getTypes(),
                              minMuleVersion,
                              runtimeExtensionModel.getVersion(),
                              new XmlDslModel(runtimeExtensionModel.getXmlDslModel().getXsdFileName(),
                                              runtimeExtensionModel.getXmlDslModel().getSchemaVersion(),
                                              runtimeExtensionModel.getXmlDslModel().getPrefix(),
                                              runtimeExtensionModel.getXmlDslModel().getNamespace(),
                                              runtimeExtensionModel.getXmlDslModel().getSchemaLocation()),
                              runtimeExtensionModel.getDescription(),
                              toDisplayModelTO(runtimeExtensionModel.getDisplayModel()),
                              toExternalLibraryModelsDTO(runtimeExtensionModel.getExternalLibraryModels()),
                              toConfigurationModelsDTO(runtimeExtensionModel),
                              toOperationModelsDTO(runtimeExtensionModel.getOperationModels()),
                              toConnectionProvidersDTO(runtimeExtensionModel.getConnectionProviders()),
                              toSourceModelsDTO(runtimeExtensionModel.getSourceModels()),
                              toFunctionModelsDTO(runtimeExtensionModel.getFunctionModels()),
                              toConstructModelsDTO(runtimeExtensionModel.getConstructModels()),
                              toErrorModelsDTO(runtimeExtensionModel.getErrorModels()));
  }

  private static Category toCategoryDTO(org.mule.runtime.api.meta.Category category) {
    if (category == org.mule.runtime.api.meta.Category.COMMUNITY) {
      return Category.communityCategory(category.name());
    }
    if (category == org.mule.runtime.api.meta.Category.SELECT) {
      return Category.selectCategory(category.name());
    }
    if (category == org.mule.runtime.api.meta.Category.PREMIUM) {
      return Category.premiumCategory(category.name());
    }
    if (category == org.mule.runtime.api.meta.Category.CERTIFIED) {
      return Category.certifiedCategory(category.name());
    }
    return new Category(category.name());
  }

  private List<ConfigurationModel> toConfigurationModelsDTO(org.mule.runtime.api.meta.model.ExtensionModel serializedExtensionModel) {
    return serializedExtensionModel.getConfigurationModels().stream()
        .map(configurationModel -> new ConfigurationModel(configurationModel.getName(),
                                                          configurationModel.getDescription(),
                                                          toParameterGroupModelsDTO(configurationModel
                                                              .getParameterGroupModels()),
                                                          toOperationModelsDTO(configurationModel.getOperationModels()),
                                                          toExternalLibraryModelsDTO(configurationModel
                                                              .getExternalLibraryModels()),
                                                          toSourceModelsDTO(configurationModel.getSourceModels()),
                                                          toConnectionProvidersDTO(configurationModel.getConnectionProviders()),
                                                          toDisplayModelTO(configurationModel.getDisplayModel()),
                                                          toStereotypeDTO(configurationModel.getStereotype())))
        .collect(toList());
  }

  private List<OperationModel> toOperationModelsDTO(List<org.mule.runtime.api.meta.model.operation.OperationModel> operationModels) {
    return operationModels.stream()
        .map(operationModel -> operationModelsCache.computeIfAbsent(operationModel.getName(),
                                                                    name -> toOperationModelDTO(operationModel)))
        .collect(toList());
  }

  private List<SourceModel> toSourceModelsDTO(List<org.mule.runtime.api.meta.model.source.SourceModel> sourceModels) {
    return sourceModels.stream()
        .map(sourceModel -> sourceModel == null ? null
            : sourceModelsCache.computeIfAbsent(sourceModel.getName(), name -> toSourceModelDTO(sourceModel)))
        .collect(toList());
  }

  private List<ConnectionProviderModel> toConnectionProvidersDTO(List<org.mule.runtime.api.meta.model.connection.ConnectionProviderModel> connectionProviders) {
    return connectionProviders.stream()
        .map(connectionProviderModel -> connectionProviderModelsCache.computeIfAbsent(connectionProviderModel.getName(),
                                                                                      name -> toConnectionProviderDTO(connectionProviderModel)))
        .collect(toList());
  }

  private List<FunctionModel> toFunctionModelsDTO(List<org.mule.runtime.api.meta.model.function.FunctionModel> functionModels) {
    return functionModels.stream()
        .map(functionModel -> functionModelsCache.computeIfAbsent(functionModel.getName(),
                                                                  name -> toFunctionModelDTO(functionModel)))
        .collect(toList());
  }

  private List<ConstructModel> toConstructModelsDTO(List<org.mule.runtime.api.meta.model.construct.ConstructModel> constructModels) {
    return constructModels.stream()
        .map(constructModel -> constructModelsCache.computeIfAbsent(constructModel.getName(),
                                                                    name -> toConstructModelDTO(constructModel)))
        .collect(toList());
  }

  private ErrorModel toErrorModelDTO(org.mule.runtime.api.meta.model.error.ErrorModel errorModel) {
    if (errorModel == null) {
      return null;
    }

    return new ErrorModel(errorModel.getType(), errorModel.getNamespace(), toErrorModelDTO(errorModel.getParent().orElse(null)));
  }

  private Set<ErrorModel> toErrorModelsDTO(Set<org.mule.runtime.api.meta.model.error.ErrorModel> errorModels) {
    return errorModels.stream()
        .map(errorModel -> errorModelsCache.computeIfAbsent(errorModel.getNamespace() + ":" + errorModel.getType(),
                                                            name -> toErrorModelDTO(errorModel)))
        .collect(toSet());
  }

}
