/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.collect.Lists.newArrayList;
import org.mule.runtime.app.declaration.api.ArtifactDeclaration;
import org.mule.tooling.agent.RuntimeToolingService;
import org.mule.tooling.client.api.artifact.ConnectivityTestingServiceBuilder;
import org.mule.tooling.client.api.connectivity.ConnectivityTestingService;

import com.mulesoft.agent.domain.tooling.BundleDescriptor;

import java.util.List;

/**
 * Implementation of {@link ConnectivityTestingServiceBuilder} that creates an artifact programmatically in order to allow doing
 * connectivity testing.
 *
 * @since 4.0
 */
public class TemporaryConnectivityTestingServiceBuilder implements ConnectivityTestingServiceBuilder {

  private List<BundleDescriptor> dependencies = newArrayList();
  private ArtifactDeclaration artifactDeclaration;
  private RuntimeToolingService runtimeToolingService;

  /**
   * Creates an instance of the builder.
   *
   * @param runtimeToolingService {@link RuntimeToolingService} to delegate remote tooling operations. Non null.
   */
  public TemporaryConnectivityTestingServiceBuilder(RuntimeToolingService runtimeToolingService) {
    checkNotNull(runtimeToolingService, "runtimeToolingService cannot be null");

    this.runtimeToolingService = runtimeToolingService;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ConnectivityTestingServiceBuilder addDependency(String groupId, String artifactId, String version, String classifier,
                                                         String type) {
    BundleDescriptor artifactBundle = new BundleDescriptor();
    artifactBundle.setGroupId(groupId);
    artifactBundle.setArtifactId(artifactId);
    artifactBundle.setVersion(version);
    artifactBundle.setClassifier(classifier);
    artifactBundle.setType(type);
    this.dependencies.add(artifactBundle);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ConnectivityTestingServiceBuilder setArtifactDeclaration(ArtifactDeclaration artifactDeclaration) {
    this.artifactDeclaration = artifactDeclaration;
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ConnectivityTestingService build() {
    return new TemporaryConnectivityTestingService(this.runtimeToolingService, this.dependencies, this.artifactDeclaration);
  }

}
