/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static java.util.stream.Collectors.toList;
import org.mule.tooling.client.api.descriptors.ArtifactDescriptor;
import org.mule.tooling.client.api.extension.ExtensionModelService;
import org.mule.tooling.client.api.extension.model.ExtensionModel;

import java.io.File;
import java.util.List;
import java.util.Optional;

/**
 * Implementation of {@link ExtensionModelService} that deleages to a {@link MuleRuntimeExtensionModelProvider} and builds the
 * Tooling view of a Mule Runtime {@link org.mule.runtime.api.meta.model.ExtensionModel} to
 * {@link org.mule.tooling.client.api.extension.model.ExtensionModel}.
 *
 * @since 4.0
 */
public class ToolingExtensionModelAdapter implements ExtensionModelService {

  private MuleRuntimeExtensionModelProvider muleRuntimeExtensionModelProvider;

  public ToolingExtensionModelAdapter(MuleRuntimeExtensionModelProvider muleRuntimeExtensionModelProvider) {
    this.muleRuntimeExtensionModelProvider = muleRuntimeExtensionModelProvider;
  }

  @Override
  public Optional<ExtensionModel> loadExtensionModel(ArtifactDescriptor pluginDescriptor) {
    return muleRuntimeExtensionModelProvider.getExtensionModel(pluginDescriptor)
        .map(runtimeExtensionModel -> toDTO(runtimeExtensionModel,
                                            muleRuntimeExtensionModelProvider.getMinMuleVersion(pluginDescriptor).get()));
  }

  @Override
  public Optional<ExtensionModel> loadExtensionModel(File plugin) {
    return muleRuntimeExtensionModelProvider.getExtensionModel(plugin)
        .map(runtimeExtensionModel -> toDTO(runtimeExtensionModel,
                                            muleRuntimeExtensionModelProvider.getMinMuleVersion(plugin).get()));
  }

  @Override
  public Optional<String> loadExtensionSchema(ArtifactDescriptor pluginDescriptor) {
    return muleRuntimeExtensionModelProvider.getExtensionSchema(pluginDescriptor);
  }

  @Override
  public Optional<String> loadExtensionSchema(File plugin) {
    return muleRuntimeExtensionModelProvider.getExtensionSchema(plugin);
  }

  @Override
  public List<ExtensionModel> loadMuleExtensionModels() {
    return muleRuntimeExtensionModelProvider.getRuntimeExtensionModels().stream()
        .map(runtimeExtensionModel -> toDTO(runtimeExtensionModel, runtimeExtensionModel.getVersion()))
        .collect(toList());
  }

  // TODO (gfernandes) MULE-13103
  @Override
  public Optional<org.mule.runtime.api.meta.model.ExtensionModel> internalExtensionModel(File plugin) {
    return muleRuntimeExtensionModelProvider.getExtensionModel(plugin);
  }

  // TODO (gfernandes) MULE-13103
  @Override
  public List<org.mule.runtime.api.meta.model.ExtensionModel> internalMuleExtensionModels() {
    return muleRuntimeExtensionModelProvider.getRuntimeExtensionModels();
  }

  private ExtensionModel toDTO(org.mule.runtime.api.meta.model.ExtensionModel runtimeExtensionModel, String minMuleVersion) {
    return new ExtensionModelFactory().createExtensionModel(runtimeExtensionModel, minMuleVersion);
  }

}
