/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.artifact;

import static org.mule.runtime.api.util.Preconditions.checkArgument;
import static org.mule.tooling.client.internal.utils.ExtensionModelUtils.resolveExtensionModels;
import org.mule.maven.client.api.MavenClient;
import org.mule.runtime.app.declaration.api.ArtifactDeclaration;
import org.mule.runtime.config.api.dsl.ArtifactDeclarationXmlSerializer;
import org.mule.runtime.api.app.declaration.serialization.ArtifactDeclarationJsonSerializer;
import org.mule.runtime.api.dsl.DslResolvingContext;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.tooling.client.api.artifact.declaration.ArtifactSerializationException;
import org.mule.tooling.client.api.artifact.declaration.ArtifactSerializationService;
import org.mule.tooling.client.api.artifact.declaration.request.AbstractXmlArtifactRequest;
import org.mule.tooling.client.api.artifact.declaration.request.JsonArtifactDeserializationRequest;
import org.mule.tooling.client.api.artifact.declaration.request.JsonArtifactSerializationRequest;
import org.mule.tooling.client.api.artifact.declaration.request.XmlArtifactDeserializationRequest;
import org.mule.tooling.client.api.artifact.declaration.request.XmlArtifactSerializationRequest;
import org.mule.tooling.client.internal.MuleRuntimeExtensionModelProvider;
import org.mule.tooling.client.internal.utils.ExtensionModelUtils;

import java.util.List;

/**
 * Default implementation of {@link ArtifactSerializationService}.
 *
 * @since 4.0
 */
public class DefaultArtifactSerializationService implements ArtifactSerializationService {

  private final List<ExtensionModel> muleModels;
  private final MuleRuntimeExtensionModelProvider muleRuntimeExtensionModelProvider;
  private final MavenClient mavenClient;

  public DefaultArtifactSerializationService(List<ExtensionModel> muleModels,
                                             MuleRuntimeExtensionModelProvider muleRuntimeExtensionModelProvider,
                                             MavenClient mavenClient) {
    this.muleModels = muleModels;
    this.muleRuntimeExtensionModelProvider = muleRuntimeExtensionModelProvider;
    this.mavenClient = mavenClient;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String toXml(XmlArtifactSerializationRequest request) {
    checkArgument(request != null, "Serialization request cannot be null");
    checkArgument(request.getArtifactDeclaration() != null,
                  "Missing ArtifactDeclaration to serialize. None was found in the given request");
    checkArgument(request.getPluginArtifactDescriptors() != null,
                  "pluginArtifactDescriptors cannot be null");

    try {
      return ArtifactDeclarationXmlSerializer.getDefault(createDslResolvingContext(request))
          .serialize(request.getArtifactDeclaration());
    } catch (Exception e) {
      throw new ArtifactSerializationException("An error occurred while serializing the ArtifactDeclaration", e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ArtifactDeclaration fromXml(XmlArtifactDeserializationRequest request) {
    checkArgument(request != null, "Deserialization request cannot be null");
    checkArgument(request.getPluginArtifactDescriptors() != null,
                  "pluginArtifactDescriptors cannot be null");
    checkArgument(request.getArtifactSource() != null,
                  "Missing ArtifactSource to deserialize. None was found in the given request");

    try {
      return ArtifactDeclarationXmlSerializer.getDefault(createDslResolvingContext(request))
          .deserialize(request.getArtifactSource());
    } catch (Exception e) {
      throw new ArtifactSerializationException("An error occurred while deserializing the ArtifactDeclaration", e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String toJson(JsonArtifactSerializationRequest request) {
    checkArgument(request != null, "Serialization request cannot be null");
    checkArgument(request.getArtifactDeclaration() != null,
                  "Missing ArtifactDeclaration to serialize. None was found in the given request");

    try {
      return ArtifactDeclarationJsonSerializer.getDefault(request.isPrettyPrint()).serialize(request.getArtifactDeclaration());
    } catch (Exception e) {
      throw new ArtifactSerializationException("An error occurred while serializing the ArtifactDeclaration", e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ArtifactDeclaration fromJson(JsonArtifactDeserializationRequest request) {
    checkArgument(request != null, "Serialization request cannot be null");
    checkArgument(request.getArtifactSource() != null && !request.getArtifactSource().trim().isEmpty(),
                  "Missing ArtifactDeclaration source to deserialize. None was found in the given request");

    try {
      return ArtifactDeclarationJsonSerializer.getDefault(false).deserialize(request.getArtifactSource());
    } catch (Exception e) {
      throw new ArtifactSerializationException("An error occurred while deserializing the ArtifactDeclaration", e);
    }
  }

  private DslResolvingContext createDslResolvingContext(AbstractXmlArtifactRequest request) {
    List<ExtensionModel> plugins = resolveExtensionModels(mavenClient,
                                                          muleRuntimeExtensionModelProvider,
                                                          request.getPluginArtifactDescriptors());

    return ExtensionModelUtils.createDslResolvingContext(muleModels, plugins);
  }

}
