/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.dsl;

import static java.lang.String.format;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static org.mule.runtime.api.util.Preconditions.checkArgument;
import static org.mule.tooling.client.internal.utils.ExtensionModelUtils.find;
import org.mule.maven.client.api.MavenClient;
import org.mule.metadata.api.model.MetadataType;
import org.mule.runtime.api.dsl.DslResolvingContext;
import org.mule.runtime.api.meta.NamedObject;
import org.mule.tooling.client.api.artifact.dsl.DslElementSyntax;
import org.mule.tooling.client.api.artifact.dsl.DslSyntaxResolver;
import org.mule.tooling.client.api.artifact.dsl.DslSyntaxResolverFactory;
import org.mule.tooling.client.api.artifact.dsl.DslSyntaxResolverService;
import org.mule.tooling.client.api.artifact.dsl.request.DslSyntaxResolverFactoryRequest;
import org.mule.tooling.client.api.descriptors.ArtifactDescriptor;
import org.mule.tooling.client.api.extension.model.ExtensionModel;
import org.mule.tooling.client.api.extension.model.config.ConfigurationModel;
import org.mule.tooling.client.api.extension.model.connection.ConnectionProviderModel;
import org.mule.tooling.client.api.extension.model.construct.ConstructModel;
import org.mule.tooling.client.api.extension.model.operation.OperationModel;
import org.mule.tooling.client.api.extension.model.source.SourceModel;
import org.mule.tooling.client.internal.MuleRuntimeExtensionModelProvider;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;

/**
 * Default implementation of a {@link DslSyntaxResolverService}
 *
 * @since 1.0
 */
public class DefaultDslSyntaxResolverService implements DslSyntaxResolverService {

  private final DslSyntaxServiceCache dslSyntaxServiceCache;

  public DefaultDslSyntaxResolverService(MuleRuntimeExtensionModelProvider muleRuntimeExtensionModelProvider,
                                         MavenClient mavenClient) {
    dslSyntaxServiceCache = new DslSyntaxServiceCache(mavenClient, muleRuntimeExtensionModelProvider);
  }

  public DslSyntaxResolverFactory getDslSyntaxResolverFactory(DslSyntaxResolverFactoryRequest request) {
    checkArgument(request != null, "The resolution request cannot be null");
    List<ArtifactDescriptor> pluginArtifactDescriptors = request.getPluginArtifactDescriptors();
    checkArgument(pluginArtifactDescriptors != null, "pluginArtifactDescriptors cannot be null");
    checkArgument(!pluginArtifactDescriptors.isEmpty(), "At least one artifact plugin is required");

    final DefaultDslSyntaxResolverService delegate = this;
    return new DslSyntaxResolverFactory() {

      @Override
      public DslSyntaxResolver createDslResolver(ExtensionModel toolingModel) {
        return new ServiceDelegatingDslSyntaxResolver(delegate, toolingModel, pluginArtifactDescriptors);
      }

      @Override
      public void dispose() {
        dslSyntaxServiceCache.invalidate(pluginArtifactDescriptors);
      }
    };
  }

  /**
   * When the tooling client is being dispose, {@code this} service should be disposed to, so all the client resources are freed.
   */
  @Override
  public void dispose() {
    this.dslSyntaxServiceCache.invalidate();
  }

  public void invalidate(List<ArtifactDescriptor> pluginArtifactDescriptors, ExtensionModel toolingModel) {
    dslSyntaxServiceCache.invalidate(pluginArtifactDescriptors, toolingModel);
  }

  public DslElementSyntax resolve(SourceModel component, ExtensionModel toolingModel,
                                  List<ArtifactDescriptor> pluginArtifactDescriptors) {
    return doResolve(extension -> find(extension, component), toolingModel, pluginArtifactDescriptors);
  }

  public DslElementSyntax resolve(OperationModel component, ExtensionModel toolingModel,
                                  List<ArtifactDescriptor> pluginArtifactDescriptors) {
    return doResolve(extension -> find(extension, component), toolingModel, pluginArtifactDescriptors);
  }

  public DslElementSyntax resolve(ConstructModel component, ExtensionModel toolingModel,
                                  List<ArtifactDescriptor> pluginArtifactDescriptors) {
    return doResolve(extension -> find(extension, component), toolingModel, pluginArtifactDescriptors);
  }

  public DslElementSyntax resolve(ConnectionProviderModel component, ExtensionModel toolingModel,
                                  List<ArtifactDescriptor> pluginArtifactDescriptors) {
    return doResolve(extension -> find(extension, component), toolingModel, pluginArtifactDescriptors);
  }

  public DslElementSyntax resolve(ConfigurationModel component, ExtensionModel toolingModel,
                                  List<ArtifactDescriptor> pluginArtifactDescriptors) {
    return doResolve(extension -> find(extension, component), toolingModel, pluginArtifactDescriptors);
  }

  Optional<DslElementSyntax> resolve(MetadataType type,
                                     ExtensionModel toolingModel,
                                     List<ArtifactDescriptor> pluginArtifactDescriptors) {
    final DslResolvingContext context = dslSyntaxServiceCache.getContext(pluginArtifactDescriptors);

    return dslSyntaxServiceCache.getResolver(findExtension(toolingModel, context), context)
        .resolve(type)
        .map(this::toDslDto);
  }

  private DslElementSyntax doResolve(Function<org.mule.runtime.api.meta.model.ExtensionModel, NamedObject> componentProvider,
                                     ExtensionModel toolingModel,
                                     List<ArtifactDescriptor> pluginArtifactDescriptors) {
    final DslResolvingContext context = dslSyntaxServiceCache.getContext(pluginArtifactDescriptors);
    final org.mule.runtime.api.meta.model.ExtensionModel extension = findExtension(toolingModel, context);

    return toDslDto(dslSyntaxServiceCache.getResolver(extension, context)
        .resolve(componentProvider.apply(extension)));
  }

  private org.mule.runtime.api.meta.model.ExtensionModel findExtension(ExtensionModel toolingExtensionModel,
                                                                       DslResolvingContext resolvingContext) {
    return resolvingContext.getExtension(toolingExtensionModel.getName())
        .orElseThrow(() -> new IllegalArgumentException(
                                                        format("Extension with name [%s] not found in the current context. Available extensions are %s",
                                                               resolvingContext.getExtensions().stream().map(e -> e.getName())
                                                                   .collect(toList()))));
  }

  private DslElementSyntax toDslDto(org.mule.runtime.extension.api.dsl.syntax.DslElementSyntax origin) {
    return new DslElementSyntax(origin.getAttributeName(),
                                origin.getElementName(),
                                origin.getPrefix(),
                                origin.getNamespace(),
                                origin.isWrapped(),
                                origin.supportsAttributeDeclaration(),
                                origin.supportsChildDeclaration(),
                                origin.supportsTopLevelDeclaration(),
                                origin.requiresConfig(),
                                origin.getGenerics().entrySet().stream()
                                    .collect(toMap(Map.Entry::getKey, e -> toDslDto(e.getValue()))),
                                origin.getContainedElementsByName().entrySet().stream()
                                    .collect(toMap(Map.Entry::getKey, e -> toDslDto(e.getValue()))));
  }

}
