/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.dsl;

import static java.lang.Integer.valueOf;
import static java.lang.String.format;
import static java.lang.System.getProperty;
import static org.mule.tooling.client.internal.utils.ExtensionModelUtils.createDslResolvingContext;
import static org.mule.tooling.client.internal.utils.ExtensionModelUtils.resolveExtensionModels;
import static org.mule.tooling.client.internal.utils.ServiceUtils.executeHandlingException;
import org.mule.maven.client.api.MavenClient;
import org.mule.runtime.api.dsl.DslResolvingContext;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.util.Preconditions;
import org.mule.runtime.extension.api.dsl.syntax.resolver.DslSyntaxResolver;
import org.mule.tooling.client.api.descriptors.ArtifactDescriptor;
import org.mule.tooling.client.internal.MuleRuntimeExtensionModelProvider;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;

import java.util.List;
import java.util.concurrent.TimeUnit;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Cache for {@link ExtensionModel}s
 */
public class DslSyntaxServiceCache {

  private static final String TOOLING_CLIENT_EXTENSION_MODEL_SERVICE_CACHE_CACHE =
      "tooling.client.DslSyntaxServiceCache.cache";
  private static final Logger LOGGER = LoggerFactory.getLogger(DslSyntaxServiceCache.class);

  private final LoadingCache<List<ArtifactDescriptor>, DslResolvingContext> dslContextCache;
  private final Cache<ExtensionModel, DslSyntaxResolver> dslResolverCache;

  public DslSyntaxServiceCache(MavenClient mavenClient, MuleRuntimeExtensionModelProvider muleRuntimeExtensionModelProvider) {
    if (LOGGER.isDebugEnabled()) {
      LOGGER.info("Initialising DSL Syntax cache");
    }

    dslContextCache = CacheBuilder.newBuilder()
        .maximumSize(cacheMaximumSize())
        .expireAfterWrite(10, TimeUnit.MINUTES)
        .build(new CacheLoader<List<ArtifactDescriptor>, DslResolvingContext>() {

          @Override
          public DslResolvingContext load(List<ArtifactDescriptor> artifacts) throws Exception {
            return createDslResolvingContext(muleRuntimeExtensionModelProvider.getRuntimeExtensionModels(),
                                             resolveExtensionModels(mavenClient,
                                                                    muleRuntimeExtensionModelProvider,
                                                                    artifacts));
          }
        });

    dslResolverCache = CacheBuilder.newBuilder()
        .maximumSize(cacheMaximumSize())
        .expireAfterWrite(10, TimeUnit.MINUTES)
        .weakKeys()
        .build();
  }

  private Integer cacheMaximumSize() {
    Integer cacheSize = valueOf(getProperty(TOOLING_CLIENT_EXTENSION_MODEL_SERVICE_CACHE_CACHE, "100"));
    Preconditions.checkArgument(cacheSize > 0,
                                format("Wrong value %d provided in system property %s, the cache cannot be less that zero",
                                       cacheSize, TOOLING_CLIENT_EXTENSION_MODEL_SERVICE_CACHE_CACHE));
    return cacheSize;
  }

  public DslResolvingContext getContext(List<ArtifactDescriptor> artifactDescriptors) {
    return executeHandlingException(() -> dslContextCache.get(artifactDescriptors));
  }

  public DslSyntaxResolver getResolver(ExtensionModel extensionModel, DslResolvingContext context) {
    return executeHandlingException(
                                    () -> dslResolverCache.get(extensionModel,
                                                               () -> DslSyntaxResolver.getDefault(extensionModel, context)));
  }

  public void invalidate(List<ArtifactDescriptor> plugins) {
    if (LOGGER.isDebugEnabled()) {
      LOGGER.debug("Invalidating DSL Context cache");
    }
    try {
      dslContextCache.get(plugins)
          .getExtensions()
          .forEach(dslResolverCache::invalidate);

      dslContextCache.invalidate(plugins);
    } catch (Exception e) {
      LOGGER.error("Failed to invalidate the DSL Context cache: " + e.getMessage(), e);
      invalidate();
    }
  }

  public void invalidate(List<ArtifactDescriptor> plugins,
                         org.mule.tooling.client.api.extension.model.ExtensionModel toolingModel) {
    if (LOGGER.isDebugEnabled()) {
      LOGGER.debug("Invalidating DSL Resolvers cache for extension: " + toolingModel.getName());
    }

    try {
      dslContextCache.get(plugins)
          .getExtension(toolingModel.getName())
          .ifPresent(dslResolverCache::invalidate);
    } catch (Exception e) {
      LOGGER.error("Failed to invalidate the DSL Resolvers cache: " + e.getMessage(), e);
      invalidate();
    }
  }

  public void invalidate() {
    if (LOGGER.isDebugEnabled()) {
      LOGGER.debug("Invalidating DSL Syntax cache");
    }
    try {
      this.dslContextCache.invalidateAll();
      this.dslResolverCache.invalidateAll();
    } catch (Exception e) {
      LOGGER.error("Failed to invalidate the DSL Syntax cache: " + e.getMessage(), e);
    }
  }

}
