/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal;

import static java.lang.Integer.valueOf;
import static java.lang.System.getProperty;
import static java.util.Collections.emptyList;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.dsl.api.component.ComponentBuildingDefinition;
import org.mule.runtime.dsl.api.component.ComponentBuildingDefinitionProvider;
import org.mule.runtime.module.extension.internal.config.ExtensionBuildingDefinitionProvider;

import com.google.common.base.Throwables;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.collect.ImmutableList;
import com.google.common.util.concurrent.UncheckedExecutionException;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ExecutionException;

/**
 * {@link ComponentBuildingDefinitionLoader} that caches the {@link ComponentBuildingDefinition}s created by
 * {@link ExtensionBuildingDefinitionProvider} since those are too complex to compute.
 */
public class CachedComponentBuildingDefinitionProviderLoader implements ComponentBuildingDefinitionLoader {

  private Cache<ExtensionModelKey, List<ComponentBuildingDefinition>> cache = CacheBuilder.newBuilder()
      .maximumSize(valueOf(getProperty("tooling.client.ComponentBuildingDefinitionProviderLoader.cache", "100")))
      .build();

  @Override
  public List<ComponentBuildingDefinition> loadComponentBuildingDefinitions(ComponentBuildingDefinitionProvider componentBuildingDefinitionProvider,
                                                                            Set<ExtensionModel> extensionModels) {

    if (!(componentBuildingDefinitionProvider instanceof ExtensionBuildingDefinitionProvider)) {
      componentBuildingDefinitionProvider.init();
      return componentBuildingDefinitionProvider.getComponentBuildingDefinitions();
    } else {
      try {
        ImmutableList.Builder<ComponentBuildingDefinition> listBuilder = ImmutableList.builder();
        Map<String, List<ComponentBuildingDefinition>> splittedProviders = new HashMap<>();
        for (ExtensionModel extensionModel : extensionModels) {
          listBuilder.addAll(cache.get(new ExtensionModelKey(extensionModel.getName(), extensionModel.getVersion()), () -> {
            if (splittedProviders.containsKey(extensionModel.getXmlDslModel().getPrefix())) {
              return splittedProviders.get(extensionModel.getXmlDslModel().getPrefix());
            }
            List<ComponentBuildingDefinition> componentBuildingDefinitions =
                retrieveComponentBuildingDefinitions(componentBuildingDefinitionProvider, extensionModels);
            componentBuildingDefinitions.forEach(definition -> {
              String definitionNamespace = definition.getComponentIdentifier().getNamespace();
              List<ComponentBuildingDefinition> definitions = splittedProviders.get(definitionNamespace);
              if (definitions == null) {
                definitions = new ArrayList<>();
                splittedProviders.put(definitionNamespace, definitions);
              }
              definitions.add(definition);
            });
            List<ComponentBuildingDefinition> extensionModelDefinitons =
                splittedProviders.get(extensionModel.getXmlDslModel().getPrefix());
            return extensionModelDefinitons != null ? extensionModelDefinitons : emptyList();
          }));
        }
        return listBuilder.build();
      } catch (ExecutionException e) {
        if (e.getCause() instanceof RuntimeException) {
          throw (RuntimeException) e.getCause();
        } else {
          throw new RuntimeException(e.getCause());
        }
      } catch (UncheckedExecutionException e) {
        Throwables.propagateIfPossible(e.getCause());
        throw Throwables.propagate(e);
      }
    }

  }

  private List<ComponentBuildingDefinition> retrieveComponentBuildingDefinitions(ComponentBuildingDefinitionProvider componentBuildingDefinitionProvider,
                                                                                 Set<ExtensionModel> extensionModels) {
    if (componentBuildingDefinitionProvider instanceof ExtensionBuildingDefinitionProvider) {
      ((ExtensionBuildingDefinitionProvider) componentBuildingDefinitionProvider)
          .setExtensionModels(extensionModels);
    }
    componentBuildingDefinitionProvider.init();
    return componentBuildingDefinitionProvider.getComponentBuildingDefinitions();
  }

  public static class ExtensionModelKey {

    private String name;
    private String version;

    public ExtensionModelKey(String name, String version) {
      this.name = name;
      this.version = version;
    }

    @Override
    public boolean equals(Object o) {
      if (this == o) {
        return true;
      }
      if (o == null || getClass() != o.getClass()) {
        return false;
      }

      ExtensionModelKey that = (ExtensionModelKey) o;

      if (!name.equals(that.name)) {
        return false;
      }
      return version.equals(that.version);
    }

    @Override
    public int hashCode() {
      int result = name.hashCode();
      result = 31 * result + version.hashCode();
      return result;
    }
  }
}
