/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static com.google.common.base.Preconditions.checkNotNull;
import static org.mule.runtime.api.metadata.resolving.MetadataFailure.Builder.newFailure;
import static org.mule.runtime.api.metadata.resolving.MetadataResult.failure;
import static org.mule.tooling.client.internal.TimeoutMethodUtils.withTimeout;
import org.mule.runtime.api.meta.model.ComponentModel;
import org.mule.runtime.api.meta.model.operation.OperationModel;
import org.mule.runtime.api.meta.model.source.SourceModel;
import org.mule.runtime.api.metadata.MetadataKeysContainer;
import org.mule.runtime.api.metadata.descriptor.ComponentMetadataDescriptor;
import org.mule.runtime.api.metadata.resolving.MetadataResult;
import org.mule.tooling.client.api.exception.ServiceUnavailableException;
import org.mule.tooling.client.api.exception.ToolingException;
import org.mule.tooling.client.api.metadata.MetadataKeysRequest;
import org.mule.tooling.client.api.metadata.MetadataRequest;
import org.mule.tooling.client.api.metadata.MetadataService;
import org.mule.tooling.client.internal.application.Deployable;

import java.util.function.Supplier;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Default implementation of {@link MetadataService}.
 *
 * @since 4.0
 */
public class DefaultMetadataService implements MetadataService, MetadataProvider {

  private final Logger logger = LoggerFactory.getLogger(this.getClass());

  private Supplier<Deployable> deployableApplicationSupplier;

  /**
   * Creates an instance of the service.
   *
   * @param deployableApplicationSupplier {@link Supplier} for a {@link Deployable}. Non null. Should return always the same instance.
   */
  public DefaultMetadataService(Supplier<Deployable> deployableApplicationSupplier) {
    checkNotNull(deployableApplicationSupplier, "deployableApplicationSupplier cannot be null");

    this.deployableApplicationSupplier = deployableApplicationSupplier;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public MetadataResult<MetadataKeysContainer> getMetadataKeys(MetadataKeysRequest metadataKeysRequest)
      throws ServiceUnavailableException, ToolingException {
    Deployable deployable = this.deployableApplicationSupplier.get();
    if (logger.isDebugEnabled()) {
      logger.debug("Getting metadata keys for: {} on application: {}", metadataKeysRequest, deployable);
    }
    Supplier<MetadataResult<MetadataKeysContainer>> metadataResultSupplier =
        () -> (MetadataResult<MetadataKeysContainer>) deployable.getRuntimeToolingService()
            .getMetadataKeys(deployable.getApplicationId(), metadataKeysRequest.getLocation().toString());
    return withTimeout(metadataKeysRequest.getRequestTimeout(),
                       metadataResultSupplier,
                       ex -> failure(newFailure((Exception) ex).onKeys()));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public MetadataResult<ComponentMetadataDescriptor<OperationModel>> getOperationMetadata(MetadataRequest metadataRequest) {
    Deployable deployable = this.deployableApplicationSupplier.get();

    return withTimeout(metadataRequest.getRequestTimeout(), () -> getMetadata(metadataRequest, deployable,
                                                                              () -> deployable.getRuntimeToolingService()
                                                                                  .getOperationMetadata(deployable
                                                                                      .getApplicationId(),
                                                                                                        metadataRequest
                                                                                                            .getLocation()
                                                                                                            .toString())),
                       ex -> failure(newFailure((Exception) ex).onComponent()));

  }

  /**
   * {@inheritDoc}
   */
  @Override
  public MetadataResult<ComponentMetadataDescriptor<SourceModel>> getSourceMetadata(MetadataRequest metadataRequest) {

    Deployable deployable = this.deployableApplicationSupplier.get();

    return withTimeout(metadataRequest.getRequestTimeout(), () -> getMetadata(metadataRequest, deployable,
                                                                              () -> deployable.getRuntimeToolingService()
                                                                                  .getSourceMetadata(deployable
                                                                                      .getApplicationId(),
                                                                                                     metadataRequest.getLocation()
                                                                                                         .toString())),
                       ex -> failure(newFailure((Exception) ex).onComponent()));
  }

  private <T extends ComponentModel> MetadataResult<ComponentMetadataDescriptor<T>> getMetadata(MetadataRequest metadataRequest,
                                                                                                Deployable deployable,
                                                                                                Supplier<MetadataResult<ComponentMetadataDescriptor<T>>> metadataResultSupplier)
      throws ServiceUnavailableException, ToolingException {
    if (logger.isDebugEnabled()) {
      logger.debug("Getting metadata for: {} on application: {}", metadataRequest, deployable);
    }

    return withTimeout(metadataRequest.getRequestTimeout(), metadataResultSupplier,
                       ex -> failure(newFailure((Exception) ex).onComponent()));
  }

}
