/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal.application;

import static com.google.common.io.Files.createTempDir;
import static java.util.Objects.requireNonNull;
import static java.util.Optional.of;
import static org.apache.commons.io.FileUtils.deleteQuietly;
import static org.apache.commons.io.FileUtils.toFile;
import static org.apache.commons.io.IOUtils.write;
import static org.mule.runtime.core.api.util.FileUtils.createTempFile;
import static org.mule.tooling.client.internal.application.ApplicationDeployerFactory.createApplicationDeployer;
import static org.slf4j.LoggerFactory.getLogger;
import org.mule.maven.client.api.MavenClient;
import org.mule.maven.client.api.model.BundleDependency;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.tooling.agent.RuntimeToolingService;
import org.mule.tooling.client.api.exception.ServiceUnavailableException;
import org.mule.tooling.client.api.extension.ExtensionModelService;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.net.URL;
import java.util.List;
import java.util.concurrent.locks.StampedLock;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import org.slf4j.Logger;

/**
 * Represents an application and allows to be deployed into Mule Runtime.
 *
 * @since 4.0
 */
public class Application {

  private static final Logger LOGGER = getLogger(Application.class);
  private final MavenClient mavenClient;
  private final ExtensionModelService extensionModelService;

  private StampedLock lock = new StampedLock();

  private Supplier<URL> applicationUrlContentSupplier;
  private Supplier<RuntimeToolingService> runtimeToolingServiceSupplier;

  protected String applicationId;

  /**
   * Creates an instance of this object.
   *
   * @param applicationUrlContentSupplier a {@link URL} {@link Supplier} for the application content that should return always the
   *        same instance.
   * @param runtimeToolingServiceSupplier a {@link RuntimeToolingService} {@link Supplier} that should return always the same
   *        instance.
   */
  public Application(Supplier<URL> applicationUrlContentSupplier, Supplier<RuntimeToolingService> runtimeToolingServiceSupplier,
                     MavenClient mavenClient, ExtensionModelService extensionModelService) {
    requireNonNull(applicationUrlContentSupplier, "applicationUrlContentSupplier cannot be null");
    requireNonNull(runtimeToolingServiceSupplier, "runtimeToolingServiceSupplier cannot be null");

    this.applicationUrlContentSupplier = applicationUrlContentSupplier;
    this.runtimeToolingServiceSupplier = runtimeToolingServiceSupplier;
    this.mavenClient = mavenClient;
    this.extensionModelService = extensionModelService;
  }

  public List<ExtensionModel> getPluginDependencies() {
    List<BundleDependency> bundleDependencies = resolveDependencies();
    List<ExtensionModel> extensionModels = bundleDependencies.stream()
        .filter(dependency -> "mule-plugin".equals(dependency.getDescriptor().getClassifier().orElse(null)))
        .map(dependency -> extensionModelService.loadExtensionModel(new File(dependency.getBundleUri())).orElse(null))
        .filter(extensionModel -> extensionModel != null)
        .collect(Collectors.toList());
    return extensionModels;
  }

  private List<BundleDependency> resolveDependencies() {
    File applicationFile = null;
    boolean deleteApplicationFile = false;
    File workDir = createTempDir();
    try {
      URL applicationUrl = applicationUrlContentSupplier.get();
      try {
        if (applicationUrl.getProtocol().equals("file")) {
          applicationFile = toFile(applicationUrl);
        } else {
          byte[] fileContent = InputStreamApplicationDeployer.readContentFromUrl(applicationUrl);
          File tempApplicationFile = createTempFile("application", "");
          try (FileOutputStream outputStream = new FileOutputStream(tempApplicationFile)) {
            write(fileContent, outputStream);
          }
          applicationFile = tempApplicationFile;
          deleteApplicationFile = true;
        }
      } catch (IOException e) {
        throw new RuntimeException(e);
      }

      return mavenClient.resolveArtifactDependencies(applicationFile, false,
                                                     of(mavenClient.getMavenConfiguration().getLocalMavenRepositoryLocation()),
                                                     of(workDir));
    } finally {
      deleteQuietly(workDir);
      if (deleteApplicationFile) {
        deleteQuietly(applicationFile);
      }
    }
  }

  /**
   * Disposes the application.
   */
  public void dispose() {
    long stamp = lock.writeLock();
    try {
      if (applicationId != null) {
        try {
          this.runtimeToolingServiceSupplier.get().disposeApplication(this.applicationId);
        } catch (ServiceUnavailableException e) {
          LOGGER.warn("Runtime Tooling Service is unavailable");
        }
        applicationId = null;
      }
    } finally {
      lock.unlockWrite(stamp);
    }

  }

  /**
   * Deploys the application and get the {@link String} applicationId.
   *
   * @return a {@link String} with the applicationId.
   */
  public Deployable deploy() {
    long stamp = lock.readLock();
    try {
      if (applicationId == null) {
        long writeStamp = lock.tryConvertToWriteLock(stamp);
        if (writeStamp == 0L) {
          lock.unlockRead(stamp);
          stamp = lock.writeLock();
        } else {
          stamp = writeStamp;
        }

        if (applicationId == null) {
          if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("It is required to deploy the application: {} in order to resolveArtifact the operation requested",
                         applicationUrlContentSupplier.get());
          }
          applicationId =
              createApplicationDeployer(applicationUrlContentSupplier.get(), runtimeToolingServiceSupplier.get()).deploy();
        }
      }
      return new ImmutableApplicationDeployable(applicationId, applicationUrlContentSupplier, runtimeToolingServiceSupplier);
    } finally {
      lock.unlock(stamp);
    }
  }

}
