/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.io.Files.createTempDir;
import static java.lang.String.format;
import static java.lang.System.setProperty;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static org.apache.commons.io.FileUtils.deleteQuietly;
import static org.mule.runtime.core.api.util.ClassUtils.withContextClassLoader;
import org.mule.maven.client.api.MavenClient;
import org.mule.maven.client.api.model.BundleDescriptor;
import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.container.api.ModuleRepository;
import org.mule.runtime.core.api.util.Pair;
import org.mule.runtime.deployment.model.api.plugin.ArtifactPluginClassLoaderFactory;
import org.mule.runtime.deployment.model.api.plugin.ArtifactPluginDescriptor;
import org.mule.runtime.deployment.model.internal.plugin.BundlePluginDependenciesResolver;
import org.mule.runtime.deployment.model.internal.tooling.ToolingPluginClassLoaderBuilder;
import org.mule.runtime.globalconfig.api.GlobalConfigLoader;
import org.mule.runtime.module.artifact.classloader.ArtifactClassLoader;
import org.mule.runtime.module.artifact.classloader.DeployableArtifactClassLoaderFactory;
import org.mule.runtime.module.artifact.classloader.MuleDeployableArtifactClassLoader;
import org.mule.runtime.module.deployment.impl.internal.artifact.ExtensionModelDiscoverer;
import org.mule.runtime.module.deployment.impl.internal.plugin.ArtifactPluginDescriptorFactory;
import org.mule.runtime.module.deployment.impl.internal.plugin.ArtifactPluginDescriptorLoader;
import org.mule.runtime.module.deployment.impl.internal.plugin.MuleExtensionModelLoaderManager;
import org.mule.runtime.module.extension.internal.loader.ExtensionModelLoaderRepository;
import org.mule.runtime.module.extension.internal.resources.MuleExtensionModelProvider;
import org.mule.tooling.client.api.descriptors.ArtifactDescriptor;
import org.mule.tooling.client.api.exception.ToolingException;
import org.mule.tooling.client.api.extension.ExtensionModelService;

import java.io.File;
import java.io.IOException;
import java.util.List;
import java.util.Optional;
import java.util.Set;

/**
 * Default implementation of {@link ExtensionModelService}.
 *
 * @since 4.0
 */
public class DefaultExtensionModelService implements InternalExtensionModelService, Initializable {

  private MavenClientHolder mavenClientHolder;
  private MavenClient mavenClient;
  private ModuleRepository moduleRepository = null;

  private ArtifactPluginDescriptorFactory artifactPluginDescriptorFactory = new ArtifactPluginDescriptorFactory();
  private ArtifactPluginDescriptorLoader artifactPluginDescriptorLoader =
      new ArtifactPluginDescriptorLoader(artifactPluginDescriptorFactory);
  private ArtifactClassLoader containerArtifactClassLoader;
  private MuleExtensionModelLoaderManager extensionModelLoaderRepository;
  private ExtensionModelDiscoverer extensionModelDiscoverer = new ExtensionModelDiscoverer();

  /**
   * Creates an instance of the service.
   */
  public DefaultExtensionModelService(MavenClientHolder mavenClientHolder, ModuleRepository moduleRepository,
                                      ArtifactClassLoader containerArtifactClassLoader) {
    this.mavenClientHolder = mavenClientHolder;
    this.moduleRepository = moduleRepository;
    this.containerArtifactClassLoader = containerArtifactClassLoader;

    extensionModelLoaderRepository =
        new MuleExtensionModelLoaderManager(containerArtifactClassLoader);
    withContextClassLoader(DefaultExtensionModelService.class.getClassLoader(), () -> {
      try {
        extensionModelLoaderRepository.start();
      } catch (MuleException e) {
        throw new ToolingException("Error while discovering ExtensionModelLoaders", e);
      }
    });
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void init() {
    mavenClient = mavenClientHolder.get();
    configureMavenProperties(mavenClient.getMavenConfiguration());
    GlobalConfigLoader.reset();
  }

  private void configureMavenProperties(MavenConfiguration mavenConfiguration) {
    setProperty("muleRuntimeConfig.maven.repositoryLocation",
                mavenConfiguration.getLocalMavenRepositoryLocation().getAbsolutePath());

    mavenConfiguration.getGlobalSettingsLocation()
        .ifPresent(globalSettingsLocation -> setProperty("muleRuntimeConfig.maven.globalSettingsLocation",
                                                         globalSettingsLocation.getAbsolutePath()));
    mavenConfiguration.getUserSettingsLocation()
        .ifPresent(userSettingsLocation -> setProperty("muleRuntimeConfig.maven.userSettingsLocation",
                                                       userSettingsLocation.getAbsolutePath()));

    mavenConfiguration.getMavenRemoteRepositories().stream().forEach(remoteRepository -> {
      String repositoryPath = format("muleRuntimeConfig.maven.repositories.%s", remoteRepository.getId());
      setProperty(repositoryPath + ".url", remoteRepository.getUrl().toString());
      remoteRepository.getAuthentication().ifPresent(authentication -> {
        setProperty(repositoryPath + ".username", authentication.getUsername());
        setProperty(repositoryPath + ".password", authentication.getPassword());
      });
    });
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ExtensionModel loadMuleCoreExtensionModel() {
    return MuleExtensionModelProvider.getExtensionModel();
  }

  private Optional<Pair<ArtifactClassLoader, ExtensionModel>> resolveExtensionModel(ArtifactPluginDescriptor artifactPluginDescriptor)
      throws IOException, MuleException {
    ToolingPluginClassLoaderBuilder builder =
        new ToolingPluginClassLoaderBuilder(new TemporaryArtifactClassLoaderFactory(),
                                            new ArtifactPluginClassLoaderFactory(moduleRepository),
                                            new BundlePluginDependenciesResolver(new ArtifactPluginDescriptorFactory()),
                                            artifactPluginDescriptor);
    builder.setParentClassLoader(containerArtifactClassLoader);
    ArtifactClassLoader pluginClassLoader = builder.build();

    Optional<ExtensionModel> extensionModel =
        getExtensionModel(artifactPluginDescriptor, pluginClassLoader, extensionModelLoaderRepository);
    if (!extensionModel.isPresent()) {
      pluginClassLoader.dispose();
      return empty();
    }
    return of(new Pair<>(pluginClassLoader, extensionModel.get()));
  }

  private Optional<ExtensionModel> getExtensionModel(ArtifactPluginDescriptor artifactPluginDescriptor,
                                                     ArtifactClassLoader pluginClassLoader,
                                                     ExtensionModelLoaderRepository extensionModelLoaderRepository) {
    Set<ExtensionModel> extensionModels =
        extensionModelDiscoverer.discoverExtensionModels(extensionModelLoaderRepository,
                                                         newArrayList(new Pair(artifactPluginDescriptor, pluginClassLoader)));
    if (extensionModels.isEmpty()) {
      return empty();
    }
    return of(extensionModels.iterator().next());
  }

  @Override
  public Optional<Pair<ArtifactClassLoader, ExtensionModel>> loadPairArtifactClassLoaderAndExtensionModel(ArtifactDescriptor pluginDescriptor) {
    BundleDescriptor.Builder builder = new BundleDescriptor.Builder();
    builder.setGroupId(pluginDescriptor.getGroupId())
        .setArtifactId(pluginDescriptor.getArtifactId())
        .setType(pluginDescriptor.getExtension())
        .setVersion(pluginDescriptor.getVersion())
        .setClassifier(pluginDescriptor.getClassifier())
        .build();
    return loadPairArtifactClassLoaderAndExtensionModel(new File(mavenClient.resolveBundleDescriptor(builder.build())
        .getBundleUri()));
  }

  @Override
  public Optional<Pair<ArtifactClassLoader, ExtensionModel>> loadPairArtifactClassLoaderAndExtensionModel(File file) {
    File tempFolder = createTempDir();
    try {
      ArtifactPluginDescriptor artifactPluginDescriptor;
      artifactPluginDescriptor = artifactPluginDescriptorLoader.load(file);
      return resolveExtensionModel(artifactPluginDescriptor);
    } catch (Exception e) {
      throw new ToolingException("Error while loading ExtensionModel for plugin: " + file.getAbsolutePath(), e);
    } finally {
      deleteQuietly(tempFolder);
    }
  }

  /**
   * Creates a class loader instance for a temporary artifact.
   */
  private class TemporaryArtifactClassLoaderFactory
      implements DeployableArtifactClassLoaderFactory<org.mule.runtime.module.artifact.descriptor.ArtifactDescriptor> {

    /**
     * {@inheritDoc}
     */
    @Override
    public ArtifactClassLoader create(String artifactId, ArtifactClassLoader parent,
                                      org.mule.runtime.module.artifact.descriptor.ArtifactDescriptor descriptor,
                                      List<ArtifactClassLoader> artifactPluginClassLoaders) {
      return new MuleDeployableArtifactClassLoader(artifactId, descriptor, descriptor.getClassLoaderModel().getUrls(),
                                                   parent.getClassLoader(),
                                                   parent.getClassLoaderLookupPolicy(), artifactPluginClassLoaders);
    }

  }

}
