/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static com.google.common.base.Preconditions.checkNotNull;
import org.mule.maven.client.api.MavenClient;
import org.mule.tooling.agent.RuntimeToolingService;
import org.mule.tooling.client.api.ToolingRuntimeClient;
import org.mule.tooling.client.api.artifact.ConnectivityTestingServiceBuilder;
import org.mule.tooling.client.api.artifact.ToolingArtifact;
import org.mule.tooling.client.api.artifact.declaration.ArtifactSerializationService;
import org.mule.tooling.client.api.configuration.agent.AgentConfiguration;
import org.mule.tooling.client.api.exception.MissingToolingConfigurationException;
import org.mule.tooling.client.api.extension.ExtensionModelService;
import org.mule.tooling.client.api.tryit.TryItService;
import org.mule.tooling.client.internal.artifact.DefaultArtifactSerializationService;

import java.net.URL;
import java.util.Optional;
import java.util.function.Supplier;

/**
 * Implementation of a {@link ToolingRuntimeClient}. It uses SPI to look up for {@link RuntimeToolingService} implementation to
 * resolve tooling operations.
 *
 * @since 4.0
 */
class DefaultToolingRuntimeClient implements ToolingRuntimeClient {

  private final MavenClient mavenClient;
  private final ComponentBuildingDefinitionLoader componentBuildingDefinitionLoader;
  private AgentConfiguration agentConfiguration;
  private Supplier<RuntimeToolingService> runtimeToolingServiceSupplier;
  private RuntimeToolingService runtimeToolingService;

  private ExtensionModelService extensionModelService;
  private ArtifactSerializationService artifactSerializationService;
  private TryItService tryItService;

  /**
   * Creates an instance of the client without configuration.
   *
   * @param runtimeToolingServiceSupplier {@link Supplier} for {@link RuntimeToolingService} to resolve operations through the
   *        Mule Runtime Agent. Not null.
   * @param mavenClient maven client
   * @param agentConfiguration agent configuration
   * @param extensionModelService service to load {@link org.mule.runtime.api.meta.model.ExtensionModel}s
   * @param componentBuildingDefinitionLoader loader for {@link org.mule.runtime.dsl.api.component.ComponentBuildingDefinition}s
   */
  DefaultToolingRuntimeClient(Supplier<RuntimeToolingService> runtimeToolingServiceSupplier,
                              MavenClient mavenClient,
                              Optional<AgentConfiguration> agentConfiguration,
                              ExtensionModelService extensionModelService,
                              ComponentBuildingDefinitionLoader componentBuildingDefinitionLoader) {
    checkNotNull(runtimeToolingServiceSupplier, "runtimeToolingServiceSupplier cannot be null");
    checkNotNull(mavenClient, "aetherDependencyResolver cannot be null");
    checkNotNull(agentConfiguration, "agentConfiguration cannot be null");

    this.mavenClient = mavenClient;
    this.runtimeToolingServiceSupplier = runtimeToolingServiceSupplier;
    this.agentConfiguration = agentConfiguration.orElse(null);
    this.componentBuildingDefinitionLoader = componentBuildingDefinitionLoader;

    // Services
    this.extensionModelService = extensionModelService;
    this.artifactSerializationService =
        new DefaultArtifactSerializationService(extensionModelService.loadMuleCoreExtensionModel(), extensionModelService,
                                                mavenClient);
    this.tryItService = new DefaultTryItService(this::configurableRuntimeToolingServiceSupplier);

    initServices();
  }

  private void initServices() {
    initIfNeeded(extensionModelService);
    initIfNeeded(artifactSerializationService);
    initIfNeeded(tryItService);
  }

  private void initIfNeeded(Object service) {
    if (service instanceof Initializable) {
      ((Initializable) service).init();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ToolingArtifact newToolingArtifact(Supplier<URL> applicationUrlSupplier) {
    return new DefaultToolingArtifact(applicationUrlSupplier, this::configurableRuntimeToolingServiceSupplier, mavenClient,
                                      extensionModelService, componentBuildingDefinitionLoader);
  }

  @Override
  public ToolingArtifact fetchToolingArtifact(String applicationId, Supplier<URL> applicationUrlSupplier) {
    return new DefaultToolingArtifact(applicationId, applicationUrlSupplier, this::configurableRuntimeToolingServiceSupplier,
                                      mavenClient, extensionModelService, componentBuildingDefinitionLoader);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public TryItService tryItService() {
    return tryItService;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ConnectivityTestingServiceBuilder newConnectivityTestingService() {
    return new TemporaryConnectivityTestingServiceBuilder(configurableRuntimeToolingServiceSupplier());
  }

  private RuntimeToolingService configurableRuntimeToolingServiceSupplier() {

    if (runtimeToolingService != null) {
      return runtimeToolingService;
    }

    if (this.agentConfiguration == null) {
      throw new MissingToolingConfigurationException("Tooling Client has to be configure with a configuration to access REST Tooling API in order to resolve operations");
    }
    this.runtimeToolingService = this.runtimeToolingServiceSupplier.get();
    this.runtimeToolingService.setToolingApiUrl(this.agentConfiguration.getToolingApiURLSupplier().get(),
                                                this.agentConfiguration.getDefaultConnectTimeout(),
                                                this.agentConfiguration.getDefaultReadTimeout(),
                                                this.agentConfiguration.getSSLContext());

    return this.runtimeToolingService;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ExtensionModelService extensionModelService() {
    return extensionModelService;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ArtifactSerializationService artifactSerializationService() {
    return artifactSerializationService;
  }

}
