/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.artifact;

import static com.google.common.collect.ImmutableSet.copyOf;
import static java.util.Optional.ofNullable;
import static java.util.stream.Collectors.toMap;
import org.mule.metadata.api.model.ObjectType;
import org.mule.runtime.api.dsl.DslResolvingContext;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.meta.type.TypeCatalog;

import com.google.common.collect.ImmutableCollection;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;

import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

/**
 * A {@link DslResolvingContext} implementation that allows to expand an original
 * {@link DslResolvingContext context} with an extra set of {@link ExtensionModel extensions}.
 *
 * @since 4.0
 */
public class ExpandedDslResolvingContext implements DslResolvingContext {

  private final TypeCatalog typeCatalog;
  private final DslResolvingContext delegate;
  private final Map<String, ExtensionModel> extensions;

  public ExpandedDslResolvingContext(DslResolvingContext delegate, Set<ExtensionModel> addedExtensions) {
    this.delegate = delegate;
    this.extensions = addedExtensions.stream()
        .filter(e -> !delegate.getExtension(e.getName()).isPresent())
        .collect(toMap(ExtensionModel::getName, e -> e, (u, v) -> v, LinkedHashMap::new));

    this.typeCatalog = new DelegatingTypeCatalog(delegate.getTypeCatalog(), copyOf(this.extensions.values()));
  }

  @Override
  public Optional<ExtensionModel> getExtension(String name) {
    return ofNullable(delegate.getExtension(name).orElseGet(() -> extensions.get(name)));
  }

  @Override
  public Set<ExtensionModel> getExtensions() {
    return ImmutableSet.<ExtensionModel>builder().addAll(delegate.getExtensions()).addAll(extensions.values()).build();
  }

  @Override
  public TypeCatalog getTypeCatalog() {
    return typeCatalog;
  }

  private static final class DelegatingTypeCatalog implements TypeCatalog {

    private final TypeCatalog originalCatalog;
    private final TypeCatalog extendingCatalog;

    public DelegatingTypeCatalog(TypeCatalog delegate, Set<ExtensionModel> addedExtensions) {
      this.originalCatalog = delegate;
      this.extendingCatalog = TypeCatalog.getDefault(addedExtensions);
    }

    @Override
    public Optional<ObjectType> getType(String typeId) {
      Optional<ObjectType> type = extendingCatalog.getType(typeId);
      return type.isPresent() ? type : originalCatalog.getType(typeId);
    }

    @Override
    public Collection<ObjectType> getTypes() {
      return mergeAvoidingDuplicates(ImmutableList.<ObjectType>builder(),
                                     originalCatalog.getTypes(),
                                     extendingCatalog.getTypes());
    }

    @Override
    public Set<ObjectType> getSubTypes(ObjectType type) {
      return mergeAvoidingDuplicates(ImmutableSet.<ObjectType>builder(),
                                     originalCatalog.getSubTypes(type),
                                     extendingCatalog.getSubTypes(type));
    }

    @Override
    public Set<ObjectType> getSuperTypes(ObjectType type) {
      return mergeAvoidingDuplicates(ImmutableSet.<ObjectType>builder(),
                                     originalCatalog.getSuperTypes(type),
                                     extendingCatalog.getSuperTypes(type));
    }

    @Override
    public Collection<ObjectType> getAllBaseTypes() {
      return mergeAvoidingDuplicates(ImmutableList.<ObjectType>builder(),
                                     originalCatalog.getAllBaseTypes(),
                                     extendingCatalog.getAllBaseTypes());
    }

    @Override
    public Collection<ObjectType> getAllSubTypes() {
      return mergeAvoidingDuplicates(ImmutableList.<ObjectType>builder(),
                                     originalCatalog.getAllSubTypes(),
                                     extendingCatalog.getAllSubTypes());
    }

    @Override
    public boolean containsBaseType(ObjectType type) {
      return extendingCatalog.containsBaseType(type) || originalCatalog.containsBaseType(type);
    }

    @Override
    public Collection<ObjectType> getExtensionTypes(String extensionName) {
      return mergeAvoidingDuplicates(ImmutableList.<ObjectType>builder(),
                                     originalCatalog.getExtensionTypes(extensionName),
                                     extendingCatalog.getExtensionTypes(extensionName));
    }

    @Override
    public Optional<String> getDeclaringExtension(String typeId) {
      Optional<String> extension = extendingCatalog.getDeclaringExtension(typeId);
      return extension.isPresent() ? extension : originalCatalog.getDeclaringExtension(typeId);
    }

    private <T extends Collection> T mergeAvoidingDuplicates(final ImmutableCollection.Builder builder,
                                                             final Collection first, final Collection second) {
      builder.addAll(first);
      second.stream()
          .filter(o -> !first.contains(o))
          .forEach(builder::add);

      return (T) builder.build();
    }

  }
}
