/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.dataweave;

import static org.mule.tooling.client.api.dataweave.DataWeavePreviewResponse.builder;
import org.mule.runtime.api.el.BindingContext;
import org.mule.runtime.api.metadata.DataType;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.api.streaming.bytes.CursorStream;
import org.mule.runtime.api.streaming.bytes.CursorStreamProvider;
import org.mule.tooling.client.api.dataweave.DataWeavePreviewRequest;
import org.mule.tooling.client.api.dataweave.DataWeavePreviewResponse;
import org.mule.tooling.event.model.EventModel;
import org.mule.tooling.event.model.MessageModel;
import org.mule.tooling.event.model.TypedValueModel;
import org.mule.weave.v2.el.WeaveExpressionLanguage;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;

import java.io.IOException;
import java.nio.charset.Charset;
import java.util.HashMap;
import java.util.Map;

import org.apache.commons.io.IOUtils;


public class LocalRunner implements IDataWeaveRunner {

  private static final String APPLICATION_JAVA = "application/java";
  private static final String DEFAULT_ATTRIBUTES_MEDIA_TYPE = "application/java";

  private static final String VARIABLES_ID = "variables";
  private static final String ATTRIBUTES_ID = "attributes";
  private static final String PAYLOAD_ID = "payload";

  /**
   * {@inheritDoc}
   */
  @Override
  public DataWeavePreviewResponse execute(DataWeavePreviewRequest parameters) {
    ClassLoader currentClassLoader = Thread.currentThread().getContextClassLoader();
    Thread.currentThread().setContextClassLoader(this.getClass().getClassLoader());
    try {
      WeaveExpressionLanguage executor = new WeaveExpressionLanguage();
      TypedValue response = executor.evaluate(getScript(parameters), buildContextFromParameters(parameters));

      return builder().result(getResult(response)).mediaType(getMediaType(response)).valid(true).build();
    } catch (Exception e) {
      return builder().errorMessage(e.getMessage()).build();
    } finally {
      Thread.currentThread().setContextClassLoader(currentClassLoader);
    }
  }

  private String getMediaType(TypedValue evaluate) {
    DataType dataType = evaluate.getDataType();
    MediaType mediaType = dataType.getMediaType();
    return mediaType.withoutParameters().toString();
  }

  private String getResult(TypedValue result) {
    try {
      final Object value = result.getValue();
      if (value instanceof CursorStreamProvider) {
        CursorStreamProvider streamProvider = (CursorStreamProvider) result.getValue();
        Charset encoding = result.getDataType().getMediaType().getCharset().orElseGet(Charset::defaultCharset);
        CursorStream cursorStream = streamProvider.openCursor();
        String content = IOUtils.toString(cursorStream, encoding);
        streamProvider.close();
        return content;
      } else {
        Gson gson = new GsonBuilder().setPrettyPrinting().create();
        return gson.toJson(value);
      }
    } catch (IOException e) {
      throw new RuntimeException("DataWeave :: Error consuming DataWeave result", e);
    }
  }

  private BindingContext buildContextFromParameters(DataWeavePreviewRequest parameters) throws IOException {
    BindingContext.Builder bindingBuilder = BindingContext.builder();

    final EventModel event = getEvent(parameters);
    if (event != null) {
      manageVariables(bindingBuilder, event);
      manageMessage(bindingBuilder, event);
    }

    return bindingBuilder.build();
  }

  private void manageMessage(BindingContext.Builder bindingBuilder, EventModel event) throws IOException {
    final MessageModel message = event.getMessage();
    if (message != null) {
      if (message.getAttributes() != null) {
        bindingBuilder.addBinding(ATTRIBUTES_ID, asTypedValue(message.getAttributes()));
      }

      if (message.getPayload() != null) {
        bindingBuilder.addBinding(PAYLOAD_ID, asTypedValue(message.getPayload()));
      }
    }
  }

  private void manageVariables(BindingContext.Builder bindingBuilder, EventModel event) throws IOException {
    if (event.getVariables() != null && !event.getVariables().isEmpty()) {
      Map<String, TypedValue> variables = new HashMap<>();
      for (Map.Entry<String, TypedValueModel> pair : event.getVariables().entrySet()) {
        variables.put(pair.getKey(), asTypedValue(pair.getValue()));
      }
      DataType variablesDataType = DataType.builder().type(Map.class).mediaType(DEFAULT_ATTRIBUTES_MEDIA_TYPE).build();
      bindingBuilder.addBinding(VARIABLES_ID, new TypedValue<>(variables, variablesDataType));
    }
  }

  private TypedValue<?> asTypedValue(TypedValueModel restTypedValue) throws IOException {
    String mediaType = restTypedValue.getDataType().getMediaType();
    if (APPLICATION_JAVA.equals(mediaType)) {
      throw new IllegalArgumentException("Java input not supported, serialize to DW script");
    } else {
      DataType dataType = DataType.builder().type(String.class).mediaType(mediaType).build();
      return new TypedValue<>(new String(restTypedValue.getContent(), "UTF-8"), dataType);
    }
  }

  private EventModel getEvent(DataWeavePreviewRequest parameters) {
    return parameters.getEvent();
  }

  private String getScript(DataWeavePreviewRequest parameters) {
    return parameters.getScript();
  }
}
