/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static com.google.common.io.Files.createTempDir;
import static java.util.Arrays.asList;
import static java.util.Objects.requireNonNull;
import static java.util.Optional.empty;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;
import static org.apache.commons.io.FileUtils.deleteQuietly;

import org.mule.maven.client.api.MavenClient;
import org.mule.maven.client.api.model.BundleDescriptor;
import org.mule.runtime.api.dsl.DslResolvingContext;
import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.util.LazyValue;
import org.mule.runtime.api.util.Pair;
import org.mule.runtime.core.api.extension.MuleExtensionModelProvider;
import org.mule.runtime.deployment.model.api.plugin.ArtifactPluginDescriptor;
import org.mule.runtime.deployment.model.internal.tooling.ToolingArtifactClassLoader;
import org.mule.runtime.deployment.model.internal.tooling.ToolingPluginClassLoaderBuilder;
import org.mule.runtime.extension.api.dsl.syntax.resources.spi.ExtensionSchemaGenerator;
import org.mule.runtime.module.artifact.api.classloader.ArtifactClassLoader;
import org.mule.runtime.module.deployment.impl.internal.artifact.ExtensionModelDiscoverer;
import org.mule.runtime.module.deployment.impl.internal.plugin.MuleExtensionModelLoaderManager;
import org.mule.runtime.module.extension.internal.capability.xml.schema.DefaultExtensionSchemaGenerator;
import org.mule.tooling.client.api.descriptors.ArtifactDescriptor;
import org.mule.tooling.client.api.exception.ToolingException;
import org.mule.tooling.client.api.extension.ExtensionModelService;
import com.mulesoft.mule.runtime.core.api.extension.MuleEeExtensionModelProvider;

import java.io.File;
import java.io.IOException;
import java.util.List;
import java.util.Optional;
import java.util.Set;

/**
 * Default implementation of {@link ExtensionModelService}.
 *
 * @since 4.0
 */
public class DefaultExtensionModelService implements InternalExtensionModelService {

  private ExtensionModelDiscoverer extensionModelDiscoverer = new ExtensionModelDiscoverer();
  private MavenClient mavenClient;
  private MuleArtifactResourcesRegistry muleArtifactResourcesRegistry;
  private ExtensionSchemaGenerator schemaGenerator = new DefaultExtensionSchemaGenerator();

  public DefaultExtensionModelService(MavenClient mavenClient,
                                      MuleArtifactResourcesRegistry muleArtifactResourcesRegistry) {
    requireNonNull(mavenClient, "mavenClient cannot be null");
    requireNonNull(muleArtifactResourcesRegistry, "muleArtifactResourcesRegistry cannot be null");

    this.mavenClient = mavenClient;
    this.muleArtifactResourcesRegistry = muleArtifactResourcesRegistry;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<ExtensionModel> loadMuleExtensionModels() {
    return asList(MuleExtensionModelProvider.getExtensionModel(), MuleEeExtensionModelProvider.getExtensionModel());
  }

  private Optional<LoadedExtensionInformation> resolveExtensionModel(ArtifactPluginDescriptor artifactPluginDescriptor)
      throws IOException, MuleException {
    ToolingPluginClassLoaderBuilder builder =
        new ToolingPluginClassLoaderBuilder(muleArtifactResourcesRegistry.newTemporaryArtifactClassLoaderFactory(),
                                            muleArtifactResourcesRegistry.getPluginDependenciesResolver(),
                                            artifactPluginDescriptor,
                                            muleArtifactResourcesRegistry.getRegionPluginClassLoadersFactory());
    builder.setParentClassLoader(muleArtifactResourcesRegistry.getContainerArtifactClassLoader());
    ToolingArtifactClassLoader toolingArtifactClassLoader = builder.build();

    MuleExtensionModelLoaderManager extensionModelLoaderRepository =
        new MuleExtensionModelLoaderManager(muleArtifactResourcesRegistry.getContainerArtifactClassLoader());
    extensionModelLoaderRepository.start();
    return getLoadedExtensionInformation(artifactPluginDescriptor, toolingArtifactClassLoader, extensionModelLoaderRepository);
  }

  private Optional<LoadedExtensionInformation> getLoadedExtensionInformation(ArtifactPluginDescriptor pluginDescriptor,
                                                                             ToolingArtifactClassLoader toolingArtifactCL,
                                                                             MuleExtensionModelLoaderManager loaderRepository) {
    Set<Pair<ArtifactPluginDescriptor, ExtensionModel>> descriptorsWithExtensions =
        discoverExtensions(toolingArtifactCL, loaderRepository);
    Optional<ExtensionModel> foundExtension = getExtensionModel(pluginDescriptor, descriptorsWithExtensions);
    if (foundExtension.isPresent()) {
      LazyValue<String> lazySchema = getSchema(descriptorsWithExtensions, foundExtension.get());
      return Optional.of(new LoadedExtensionInformation(foundExtension.get(), toolingArtifactCL, lazySchema));
    } else {
      toolingArtifactCL.dispose();
      return empty();
    }
  }

  private LazyValue<String> getSchema(Set<Pair<ArtifactPluginDescriptor, ExtensionModel>> descriptorsWithExtensions,
                                      ExtensionModel foundExtension) {
    Set<ExtensionModel> extensions = descriptorsWithExtensions.stream().map(Pair::getSecond).collect(toSet());
    return new LazyValue<>(() -> schemaGenerator.generate(foundExtension, DslResolvingContext.getDefault(extensions)));
  }

  private Optional<ExtensionModel> getExtensionModel(ArtifactPluginDescriptor descriptor,
                                                     Set<Pair<ArtifactPluginDescriptor, ExtensionModel>> descriptorsWithExtensions) {
    if (descriptorsWithExtensions.isEmpty()) {
      return empty();
    }
    return descriptorsWithExtensions.stream().filter(e -> e.getFirst().equals(descriptor)).map(Pair::getSecond).findFirst();
  }

  private Set<Pair<ArtifactPluginDescriptor, ExtensionModel>> discoverExtensions(ToolingArtifactClassLoader toolingArtifactCL,
                                                                                 MuleExtensionModelLoaderManager extensionModelLoaderRepository) {
    List<Pair<ArtifactPluginDescriptor, ArtifactClassLoader>> artifacts = toolingArtifactCL.getArtifactPluginClassLoaders()
        .stream()
        .map(a -> new Pair<ArtifactPluginDescriptor, ArtifactClassLoader>(a.getArtifactDescriptor(), a))
        .collect(toList());
    return extensionModelDiscoverer.discoverExtensionModels(extensionModelLoaderRepository, artifacts);
  }

  @Override
  public Optional<LoadedExtensionInformation> loadExtensionData(ArtifactDescriptor pluginDescriptor) {
    BundleDescriptor.Builder builder = new BundleDescriptor.Builder();
    builder.setGroupId(pluginDescriptor.getGroupId())
        .setArtifactId(pluginDescriptor.getArtifactId())
        .setType(pluginDescriptor.getExtension())
        .setVersion(pluginDescriptor.getVersion())
        .setClassifier(pluginDescriptor.getClassifier())
        .build();
    return loadExtensionData(new File(mavenClient.resolveBundleDescriptor(builder.build())
        .getBundleUri()));
  }

  @Override
  public Optional<LoadedExtensionInformation> loadExtensionData(File file) {
    File tempFolder = createTempDir();
    try {
      ArtifactPluginDescriptor artifactPluginDescriptor;
      artifactPluginDescriptor = muleArtifactResourcesRegistry.getArtifactPluginDescriptorLoader().load(file);
      return resolveExtensionModel(artifactPluginDescriptor);
    } catch (Exception e) {
      throw new ToolingException("Error while loading ExtensionModel for plugin: " + file.getAbsolutePath(), e);
    } finally {
      deleteQuietly(tempFolder);
    }
  }

}
