/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static java.util.Collections.emptySet;
import static java.util.stream.Collectors.toCollection;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;
import static org.mule.tooling.client.internal.CollectionUtils.copy;
import org.mule.runtime.extension.internal.property.MetadataKeyPartModelProperty;
import org.mule.tooling.client.api.extension.model.DisplayModel;
import org.mule.tooling.client.api.extension.model.ErrorModel;
import org.mule.tooling.client.api.extension.model.ExpressionSupport;
import org.mule.tooling.client.api.extension.model.ExternalLibraryModel;
import org.mule.tooling.client.api.extension.model.ExternalLibraryType;
import org.mule.tooling.client.api.extension.model.LayoutModel;
import org.mule.tooling.client.api.extension.model.OutputModel;
import org.mule.tooling.client.api.extension.model.ParameterDslConfiguration;
import org.mule.tooling.client.api.extension.model.PathModel;
import org.mule.tooling.client.api.extension.model.PathModelType;
import org.mule.tooling.client.api.extension.model.StereotypeModel;
import org.mule.tooling.client.api.extension.model.connection.ConnectionManagementType;
import org.mule.tooling.client.api.extension.model.connection.ConnectionProviderModel;
import org.mule.tooling.client.api.extension.model.construct.ConstructModel;
import org.mule.tooling.client.api.extension.model.function.FunctionModel;
import org.mule.tooling.client.api.extension.model.metadata.MetadataKeyPartModel;
import org.mule.tooling.client.api.extension.model.nested.NestableElementModel;
import org.mule.tooling.client.api.extension.model.nested.NestedChainModel;
import org.mule.tooling.client.api.extension.model.nested.NestedComponentModel;
import org.mule.tooling.client.api.extension.model.nested.NestedRouteModel;
import org.mule.tooling.client.api.extension.model.operation.OperationModel;
import org.mule.tooling.client.api.extension.model.parameter.ExclusiveParametersModel;
import org.mule.tooling.client.api.extension.model.parameter.ParameterGroupModel;
import org.mule.tooling.client.api.extension.model.parameter.ParameterModel;
import org.mule.tooling.client.api.extension.model.parameter.ParameterRole;
import org.mule.tooling.client.api.extension.model.source.SourceCallbackModel;
import org.mule.tooling.client.api.extension.model.source.SourceModel;
import org.mule.tooling.client.api.extension.model.value.ValueProviderModel;
import org.mule.tooling.client.api.location.Location;

import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Optional;
import java.util.Set;

/**
 * Utils for creating the Extension Model DTOs from the Mule-API objects
 *
 * @since 4.0
 */
public class ExtensionModelPartsFactory {

  public static Set<ExternalLibraryModel> toExternalLibraryModelsDTO(
                                                                     Set<org.mule.runtime.api.meta.model.ExternalLibraryModel> externalLibraryModels) {
    return externalLibraryModels.stream().map(externalLibraryModel -> new ExternalLibraryModel(externalLibraryModel.getName(),
                                                                                               externalLibraryModel
                                                                                                   .getDescription(),
                                                                                               externalLibraryModel
                                                                                                   .getRequiredClassName()
                                                                                                   .orElse(null),
                                                                                               externalLibraryModel
                                                                                                   .getRegexMatcher()
                                                                                                   .orElse(null),
                                                                                               toExternalLibraryTypeDTO(externalLibraryModel
                                                                                                   .getType())))
        .collect(toSet());
  }

  private static ExternalLibraryType toExternalLibraryTypeDTO(org.mule.runtime.api.meta.ExternalLibraryType type) {
    if (type == org.mule.runtime.api.meta.ExternalLibraryType.NATIVE) {
      return ExternalLibraryType.nativeExternalLibraryType(type.name());
    }
    if (type == org.mule.runtime.api.meta.ExternalLibraryType.JAR) {
      return ExternalLibraryType.jarExternalLibraryType(type.name());
    }
    if (type == org.mule.runtime.api.meta.ExternalLibraryType.DEPENDENCY) {
      return ExternalLibraryType.dependencyExternalLibraryType(type.name());
    }
    return new ExternalLibraryType(type.name());
  }


  public static ConnectionProviderModel toConnectionProviderDTO(org.mule.runtime.api.meta.model.connection.ConnectionProviderModel connectionProviderModel) {
    return new ConnectionProviderModel(connectionProviderModel.getName(),
                                       connectionProviderModel.getDescription(),
                                       toDisplayModelTO(connectionProviderModel.getDisplayModel()),
                                       toConnectionManagementTypeDTO(connectionProviderModel.getConnectionManagementType()),
                                       toParameterGroupModelsDTO(connectionProviderModel.getParameterGroupModels()),
                                       toExternalLibraryModelsDTO(connectionProviderModel.getExternalLibraryModels()),
                                       toStereotypeDTO(connectionProviderModel.getStereotype()));
  }

  private static ConnectionManagementType toConnectionManagementTypeDTO(
                                                                        org.mule.runtime.api.meta.model.connection.ConnectionManagementType connectionManagementType) {
    if (connectionManagementType == org.mule.runtime.api.meta.model.connection.ConnectionManagementType.POOLING) {
      return ConnectionManagementType.poolingConnectionManagementType(connectionManagementType.name());
    }
    if (connectionManagementType == org.mule.runtime.api.meta.model.connection.ConnectionManagementType.CACHED) {
      return ConnectionManagementType.cachedConnectionManagementType(connectionManagementType.name());
    }
    if (connectionManagementType == org.mule.runtime.api.meta.model.connection.ConnectionManagementType.NONE) {
      return ConnectionManagementType.noneConnectionManagementType(connectionManagementType.name());
    }

    return new ConnectionManagementType(connectionManagementType.name());
  }


  public static SourceModel toSourceModelDTO(org.mule.runtime.api.meta.model.source.SourceModel sourceModel) {
    if (sourceModel == null) {
      return null;
    }
    return new SourceModel(sourceModel.getName(),
                           sourceModel.getDescription(),
                           toParameterGroupModelsDTO(sourceModel.getParameterGroupModels()),
                           toDisplayModelTO(sourceModel.getDisplayModel()),
                           toOutputModelDTO(sourceModel.getOutput()),
                           toOutputModelDTO(sourceModel.getOutputAttributes()),
                           sourceModel.isTransactional(),
                           sourceModel.requiresConnection(),
                           sourceModel.supportsStreaming(),
                           toStereotypeDTO(sourceModel.getStereotype()),
                           sourceModel.hasResponse(),
                           toSourceCallbackModelDTO(sourceModel.getSuccessCallback()),
                           toSourceCallbackModelDTO(sourceModel.getErrorCallback()),
                           toSourceCallbackModelDTO(sourceModel.getTerminateCallback()),
                           toErrorModelsDTO(sourceModel.getErrorModels()),
                           toNestedComponentModelsDTO(sourceModel.getNestedComponents()));
  }

  private static List<? extends NestableElementModel> toNestedComponentModelsDTO(
                                                                                 List<? extends org.mule.runtime.api.meta.model.nested.NestableElementModel> components) {
    return components.stream().map(model -> toNestedComponentModelDTO(model)).collect(toList());
  }

  private static NestableElementModel toNestedComponentModelDTO(org.mule.runtime.api.meta.model.nested.NestableElementModel nestedModel) {

    if (nestedModel instanceof org.mule.runtime.api.meta.model.nested.NestedRouteModel) {
      return new NestedRouteModel(nestedModel.getName(),
                                  nestedModel.getDescription(),
                                  toDisplayModelTO(nestedModel.getDisplayModel()),
                                  ((org.mule.runtime.api.meta.model.nested.NestedRouteModel) nestedModel).getMinOccurs(),
                                  ((org.mule.runtime.api.meta.model.nested.NestedRouteModel) nestedModel).getMaxOccurs()
                                      .orElse(null),
                                  toParameterGroupModelsDTO(((org.mule.runtime.api.meta.model.nested.NestedRouteModel) nestedModel)
                                      .getParameterGroupModels()),
                                  toNestedComponentModelsDTO(((org.mule.runtime.api.meta.model.nested.NestedRouteModel) nestedModel)
                                      .getNestedComponents()));
    }

    if (nestedModel instanceof org.mule.runtime.api.meta.model.nested.NestedChainModel) {
      return new NestedChainModel(nestedModel.getName(),
                                  nestedModel.getDescription(),
                                  toDisplayModelTO(nestedModel.getDisplayModel()),
                                  nestedModel.isRequired(),
                                  toStereotypesDTO(((org.mule.runtime.api.meta.model.nested.NestedComponentModel) nestedModel)
                                      .getAllowedStereotypes()));
    }

    if (nestedModel instanceof org.mule.runtime.api.meta.model.nested.NestedComponentModel) {
      return new NestedComponentModel(nestedModel.getName(),
                                      nestedModel.getDescription(),
                                      toDisplayModelTO(nestedModel.getDisplayModel()),
                                      nestedModel.isRequired(),
                                      toStereotypesDTO(((org.mule.runtime.api.meta.model.nested.NestedComponentModel) nestedModel)
                                          .getAllowedStereotypes()));
    }

    return null;
  }

  private static Set<ErrorModel> toErrorModelsDTO(Set<org.mule.runtime.api.meta.model.error.ErrorModel> errorModels) {
    return errorModels.stream()
        .map(errorModel -> toErrorModelDTO(errorModel))
        .collect(toSet());
  }

  private static ErrorModel toErrorModelDTO(org.mule.runtime.api.meta.model.error.ErrorModel errorModel) {
    if (errorModel == null) {
      return null;
    }
    return new ErrorModel(errorModel.getType(), errorModel.getNamespace(),
                          toErrorModelDTO(errorModel.getParent().orElse(null)));
  }

  private static SourceCallbackModel toSourceCallbackModelDTO(Optional<org.mule.runtime.api.meta.model.source.SourceCallbackModel> sourceCallbackModelOptional) {
    return sourceCallbackModelOptional
        .map(sourceCallbackModel -> new SourceCallbackModel(sourceCallbackModel.getName(),
                                                            sourceCallbackModel.getDescription(),
                                                            toParameterGroupModelsDTO(sourceCallbackModel
                                                                .getParameterGroupModels()),
                                                            toDisplayModelTO(sourceCallbackModel.getDisplayModel())))
        .orElse(null);
  }

  private static Set<StereotypeModel> toStereotypesDTO(Collection<org.mule.runtime.api.meta.model.stereotype.StereotypeModel> stereotypes) {
    // Mule Extension Models (json written by hand) has null value for this property
    if (stereotypes == null) {
      return emptySet();
    }
    return stereotypes.stream()
        .map(stereotype -> new StereotypeModel(stereotype.getName(), stereotype.getNamespace(),
                                               stereotype.getParent().map(ExtensionModelPartsFactory::toStereotypeDTO)
                                                   .orElse(null)))
        .collect(toCollection(LinkedHashSet::new));
  }

  static StereotypeModel toStereotypeDTO(org.mule.runtime.api.meta.model.stereotype.StereotypeModel stereotype) {
    // Mule Extension Models (json written by hand) has null value for this property
    if (stereotype == null) {
      return null;
    }
    return new StereotypeModel(stereotype.getName(), stereotype.getNamespace(),
                               stereotype.getParent().map(ExtensionModelPartsFactory::toStereotypeDTO).orElse(null));
  }

  private static OutputModel toOutputModelDTO(org.mule.runtime.api.meta.model.OutputModel output) {
    return new OutputModel(output.getDescription(), output.getType(), output.hasDynamicType());
  }

  public static OperationModel toOperationModelDTO(org.mule.runtime.api.meta.model.operation.OperationModel operationModel) {
    if (operationModel == null) {
      return null;
    }

    return new OperationModel(operationModel.getName(),
                              operationModel.getDescription(),
                              toParameterGroupModelsDTO(operationModel.getParameterGroupModels()),
                              toDisplayModelTO(operationModel.getDisplayModel()),
                              operationModel.isBlocking(),
                              toErrorModelsDTO(operationModel.getErrorModels()),
                              toOutputModelDTO(operationModel.getOutput()),
                              toOutputModelDTO(operationModel.getOutputAttributes()),
                              operationModel.isTransactional(),
                              operationModel.requiresConnection(),
                              operationModel.supportsStreaming(),
                              toStereotypeDTO(operationModel.getStereotype()),
                              toNestedComponentModelsDTO(operationModel.getNestedComponents()));
  }

  public static List<ParameterGroupModel> toParameterGroupModelsDTO(List<org.mule.runtime.api.meta.model.parameter.ParameterGroupModel> parameterGroupModels) {
    return parameterGroupModels.stream()
        .map(parameterGroupModel -> new ParameterGroupModel(parameterGroupModel.getName(),
                                                            parameterGroupModel.getDescription(),
                                                            toDisplayModelTO(parameterGroupModel.getDisplayModel()),
                                                            toLayoutModelDTO(parameterGroupModel.getLayoutModel()),
                                                            toParameterModelsDTO(parameterGroupModel.getParameterModels()),
                                                            toExclusiveParameterModelsDTO(parameterGroupModel
                                                                .getExclusiveParametersModels()),
                                                            parameterGroupModel.isShowInDsl()))
        .collect(toList());
  }

  private static List<ExclusiveParametersModel> toExclusiveParameterModelsDTO(List<org.mule.runtime.api.meta.model.parameter.ExclusiveParametersModel> exclusiveParametersModels) {
    return exclusiveParametersModels.stream()
        .map(exclusiveParametersModel -> new ExclusiveParametersModel(copy(exclusiveParametersModel
            .getExclusiveParameterNames()),
                                                                      exclusiveParametersModel.isOneRequired()))
        .collect(toList());
  }

  private static List<ParameterModel> toParameterModelsDTO(List<org.mule.runtime.api.meta.model.parameter.ParameterModel> parameterModels) {
    return parameterModels.stream()
        .map(parameterModel -> new ParameterModel(parameterModel.getName(),
                                                  parameterModel.getDescription(),
                                                  toDisplayModelTO(parameterModel.getDisplayModel()),
                                                  parameterModel.getType(),
                                                  parameterModel.hasDynamicType(),
                                                  parameterModel.isRequired(),
                                                  parameterModel.isOverrideFromConfig(),
                                                  toExpressionSupportDTO(parameterModel.getExpressionSupport()),
                                                  parameterModel.getDefaultValue(),
                                                  toParameterRoleDTO(parameterModel.getRole()),
                                                  toParameterDslConfigurationDTO(parameterModel.getDslConfiguration()),
                                                  toLayoutModelDTO(parameterModel.getLayoutModel()),
                                                  toMetadataKeyPartModel(parameterModel
                                                      .getModelProperty(MetadataKeyPartModelProperty.class)),
                                                  toValueProviderModelDTO(parameterModel.getValueProviderModel()),
                                                  new ArrayList<>(toStereotypesDTO(parameterModel.getAllowedStereotypes()))))
        .collect(toList());
  }

  private static MetadataKeyPartModel toMetadataKeyPartModel(Optional<MetadataKeyPartModelProperty> modelProperty) {
    return modelProperty.map(metadataKeyPartModelProperty -> new MetadataKeyPartModel(metadataKeyPartModelProperty.getOrder()))
        .orElse(null);
  }

  private static ExpressionSupport toExpressionSupportDTO(org.mule.runtime.api.meta.ExpressionSupport expressionSupport) {
    if (expressionSupport == org.mule.runtime.api.meta.ExpressionSupport.SUPPORTED) {
      return ExpressionSupport.supportedExpressionSupport(expressionSupport.name());
    }
    if (expressionSupport == org.mule.runtime.api.meta.ExpressionSupport.NOT_SUPPORTED) {
      return ExpressionSupport.notSupportedExpressionSupport(expressionSupport.name());
    }
    if (expressionSupport == org.mule.runtime.api.meta.ExpressionSupport.REQUIRED) {
      return ExpressionSupport.requiredExpressionSupport(expressionSupport.name());
    }
    return new ExpressionSupport(expressionSupport.name());
  }

  private static ParameterDslConfiguration toParameterDslConfigurationDTO(
                                                                          org.mule.runtime.api.meta.model.ParameterDslConfiguration dslConfiguration) {
    return new ParameterDslConfiguration(dslConfiguration.allowsInlineDefinition(), dslConfiguration.allowTopLevelDefinition(),
                                         dslConfiguration.allowsReferences());
  }

  private static LayoutModel toLayoutModelDTO(Optional<org.mule.runtime.api.meta.model.display.LayoutModel> layoutModelOptional) {
    return layoutModelOptional.map(layoutModel -> new LayoutModel(layoutModel.getTabName().orElse(null),
                                                                  layoutModel.getOrder().orElse(null),
                                                                  layoutModel.isPassword(),
                                                                  layoutModel.isText(),
                                                                  layoutModel.isQuery()))
        .orElse(null);
  }

  private static ParameterRole toParameterRoleDTO(org.mule.runtime.api.meta.model.parameter.ParameterRole parameterRole) {
    if (parameterRole == org.mule.runtime.api.meta.model.parameter.ParameterRole.BEHAVIOUR) {
      return ParameterRole.behaviourParameterRole(parameterRole.name());
    }
    if (parameterRole == org.mule.runtime.api.meta.model.parameter.ParameterRole.CONTENT) {
      return ParameterRole.contentParameterRole(parameterRole.name());
    }
    if (parameterRole == org.mule.runtime.api.meta.model.parameter.ParameterRole.PRIMARY_CONTENT) {
      return ParameterRole.primaryContentParameterRole(parameterRole.name());
    }
    return new ParameterRole(parameterRole.name());
  }

  public static DisplayModel toDisplayModelTO(Optional<org.mule.runtime.api.meta.model.display.DisplayModel> displayModelOptional) {
    return displayModelOptional.map(displayModel -> new DisplayModel(displayModel.getDisplayName(), displayModel.getSummary(),
                                                                     displayModel.getExample(),
                                                                     toPathModelDTO(displayModel.getPathModel())))
        .orElse(null);
  }

  private static PathModel toPathModelDTO(Optional<org.mule.runtime.api.meta.model.display.PathModel> pathModelOptional) {
    return pathModelOptional.map(pathModel -> new PathModel(toPathModelTypeDTO(pathModel.getType()), pathModel.acceptsUrls(),
                                                            pathModel.getFileExtensions().toArray(new String[0])))
        .orElse(null);
  }

  private static PathModelType toPathModelTypeDTO(org.mule.runtime.api.meta.model.display.PathModel.Type type) {
    if (type == org.mule.runtime.api.meta.model.display.PathModel.Type.DIRECTORY) {
      return PathModelType.directoryPathModelType(type.name());
    }
    if (type == org.mule.runtime.api.meta.model.display.PathModel.Type.FILE) {
      return PathModelType.fileyPathModelType(type.name());
    }
    if (type == org.mule.runtime.api.meta.model.display.PathModel.Type.ANY) {
      return PathModelType.anyPathModelType(type.name());
    }
    return new PathModelType(type.name());
  }

  private static ValueProviderModel toValueProviderModelDTO(Optional<org.mule.runtime.api.meta.model.parameter.ValueProviderModel> valueProviderModel) {
    return valueProviderModel
        .map(model -> new ValueProviderModel(model.getActingParameters(), model.requiresConfiguration(),
                                             model.requiresConnection(), model.isOpen(), model.getPartOrder(),
                                             model.getProviderName()))
        .orElse(null);
  }

  public static Location toLocationDTO(org.mule.runtime.api.component.location.Location location) {
    if (location == null) {
      return null;
    }
    String global = location.getGlobalName();
    List<String> rest = location.getParts();

    LinkedList<String> parts = new LinkedList<>();
    parts.add(global);
    parts.addAll(rest);

    return new Location(parts);
  }

  public static FunctionModel toFunctionModelDTO(org.mule.runtime.api.meta.model.function.FunctionModel functionModel) {
    return new FunctionModel(functionModel.getName(), functionModel.getDescription(),
                             toParameterGroupModelsDTO(functionModel.getParameterGroupModels()),
                             toDisplayModelTO(functionModel.getDisplayModel()), toOutputModelDTO(functionModel.getOutput()));
  }

  public static List<ConstructModel> toConstructModelsDTO(List<org.mule.runtime.api.meta.model.construct.ConstructModel> constructModels) {
    return constructModels.stream().map(model -> toConstructModelDTO(model)).collect(toList());
  }

  public static ConstructModel toConstructModelDTO(org.mule.runtime.api.meta.model.construct.ConstructModel constructModel) {
    return new ConstructModel(constructModel.getName(), constructModel.getDescription(),
                              toParameterGroupModelsDTO(constructModel.getParameterGroupModels()),
                              toDisplayModelTO(constructModel.getDisplayModel()),
                              toStereotypeDTO(constructModel.getStereotype()),
                              toNestedComponentModelsDTO(constructModel.getNestedComponents()),
                              constructModel.allowsTopLevelDeclaration());
  }

}
