/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal;

import static java.lang.Integer.valueOf;
import static java.lang.String.format;
import static java.lang.System.getProperty;

import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.util.Preconditions;
import org.mule.runtime.module.artifact.api.classloader.ArtifactClassLoader;
import org.mule.tooling.client.api.Disposable;
import org.mule.tooling.client.api.descriptors.ArtifactDescriptor;
import com.google.common.base.Throwables;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.RemovalListener;
import com.google.common.util.concurrent.UncheckedExecutionException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import java.io.File;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.ExecutionException;

/**
 * Cache for {@link ExtensionModel}s
 */
public class ExtensionModelServiceCache implements Disposable {

  private static final String TOOLING_CLIENT_EXTENSION_MODEL_SERVICE_CACHE_CACHE =
      "tooling.client.ExtensionModelServiceCache.cache";
  private static final Logger LOGGER = LoggerFactory.getLogger(ExtensionModelServiceCache.class);

  private List<ExtensionModel> muleExtensionModels;
  private Cache<ArtifactDescriptor, Optional<LoadedExtensionInformation>> extensionModelsByArtifact =
      CacheBuilder.newBuilder()
          .maximumSize(cacheMaximumSize())
          .removalListener((RemovalListener<ArtifactDescriptor, Optional<LoadedExtensionInformation>>) notification -> notification
              .getValue()
              .ifPresent(
                         data -> closeClassLoader(
                                                  data
                                                      .getArtifactClassLoader())))
          .build();
  private Cache<File, Optional<LoadedExtensionInformation>> extensionModelsByFile =
      CacheBuilder.newBuilder()
          .maximumSize(cacheMaximumSize())
          .removalListener((RemovalListener<File, Optional<LoadedExtensionInformation>>) notification -> notification
              .getValue()
              .ifPresent(
                         data -> closeClassLoader(data
                             .getArtifactClassLoader())))
          .build();

  private Integer cacheMaximumSize() {
    Integer cacheSize = valueOf(getProperty(TOOLING_CLIENT_EXTENSION_MODEL_SERVICE_CACHE_CACHE, "100"));
    Preconditions.checkArgument(cacheSize > 0,
                                format("Wrong value %d provided in system property %s, the cache cannot be less that zero",
                                       cacheSize, TOOLING_CLIENT_EXTENSION_MODEL_SERVICE_CACHE_CACHE));
    return cacheSize;
  }

  /**
   * Loads an extension model from a {@link ArtifactDescriptor}.
   * <p>
   * It uses a cache, so it will only call the {@code extensionModelService} if there's a cache miss.
   *
   * @param artifact                the plugin descriptor
   * @param extensionModelService the service to use to load the {@link ExtensionModel} in case of a cache miss
   * @return the extension model, empty if the plugin does not have an extension model
   */
  public Optional<ExtensionModel> loadExtensionModel(ArtifactDescriptor artifact,
                                                     InternalExtensionModelService extensionModelService) {
    return executeHandlingException(() -> extensionModelsByArtifact
        .get(artifact, () -> extensionModelService.loadExtensionData(artifact))
        .map(LoadedExtensionInformation::getExtensionModel));
  }

  /**
   * Loads an extension model from a {@link File} pointing to the plugin artifact.
   * <p>
   * It uses a cache, so it will only call the {@code extensionModelService} if there's a cache miss.
   *
   * @param plugin                the plugin descriptor
   * @param extensionModelService the service to use to load the {@link ExtensionModel} in case of a cache miss
   * @return the extension model, empty if the plugin does not have an extension model
   */
  public Optional<ExtensionModel> loadExtensionModel(File plugin, InternalExtensionModelService extensionModelService) {
    return executeHandlingException(() -> extensionModelsByFile
        .get(plugin, () -> extensionModelService.loadExtensionData(plugin))
        .map(LoadedExtensionInformation::getExtensionModel));
  }

  /**
   * Loads the XML Schema for the given {@link File} pointing to the plugin artifact.
   * <p>
   * It uses a cache, so it will only call the {@code extensionModelService} if there's a cache miss.
   *
   * @param plugin                the file pointing to the plugin descriptor.
   * @param extensionModelService the service to use to load the {@link ExtensionModel} in case of a cache miss
   * @return an String with the schema content, empty if the plugin does not have an extension model.
   */
  public Optional<String> loadExtensionSchema(File plugin, InternalExtensionModelService extensionModelService) {
    return executeHandlingException(() -> extensionModelsByFile
        .get(plugin, () -> extensionModelService.loadExtensionData(plugin))
        .map(e -> e.getSchema().get()));
  }

  /**
   * Loads the XML Schema for a plugin given it's {@link ArtifactDescriptor}.
   * <p>
   * It uses a cache, so it will only call the {@code extensionModelService} if there's a cache miss.
   *
   * @param artifact              the plugin descriptor.
   * @param extensionModelService the service to use to load the {@link ExtensionModel} in case of a cache miss
   * @return an String with the schema content, empty if the plugin does not have an extension model
   */
  public Optional<String> loadExtensionSchema(ArtifactDescriptor artifact, InternalExtensionModelService extensionModelService) {
    return executeHandlingException(() -> extensionModelsByArtifact
        .get(artifact, () -> extensionModelService.loadExtensionData(artifact))
        .map(e -> e.getSchema().get()));
  }

  /**
   * Loads the core extension model.
   *
   * @param extensionModelService the service to use to load the {@link ExtensionModel} in case of a cache miss
   * @return the core extension model
   */
  public List<ExtensionModel> loadMuleExtensionModels(InternalExtensionModelService extensionModelService) {
    if (muleExtensionModels == null) {
      muleExtensionModels = extensionModelService.loadMuleExtensionModels();
    }
    return muleExtensionModels;
  }

  private void closeClassLoader(ArtifactClassLoader artifactClassLoader) {
    try {
      artifactClassLoader.dispose();
    } catch (Exception e) {
      if (LOGGER.isDebugEnabled()) {
        LOGGER.debug(e.getMessage(), e);
      }
    }
  }

  @Override
  public void dispose() {
    this.extensionModelsByFile.invalidateAll();
    this.extensionModelsByArtifact.invalidateAll();
  }

  private <T> T executeHandlingException(ResourceSupplier<T> supplier) {
    try {
      return supplier.getResource();
    } catch (ExecutionException e) {
      if (e.getCause() instanceof RuntimeException) {
        throw (RuntimeException) e.getCause();
      } else {
        throw new RuntimeException(e.getCause());
      }
    } catch (UncheckedExecutionException e) {
      Throwables.propagateIfPossible(e.getCause());
      throw Throwables.propagate(e);
    }
  }

  @FunctionalInterface
  private interface ResourceSupplier<T> {

    T getResource() throws ExecutionException;
  }
}
