/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Preconditions.checkState;
import static java.util.Optional.empty;
import org.mule.maven.client.api.MavenClient;
import org.mule.runtime.module.embedded.api.EmbeddedContainer;
import org.mule.tooling.agent.RuntimeToolingService;
import org.mule.tooling.client.api.ToolingRuntimeClient;
import org.mule.tooling.client.api.artifact.ConnectivityTestingServiceBuilder;
import org.mule.tooling.client.api.artifact.ToolingArtifact;
import org.mule.tooling.client.api.artifact.declaration.ArtifactSerializationService;
import org.mule.tooling.client.api.configuration.agent.AgentConfiguration;
import org.mule.tooling.client.api.exception.MissingToolingConfigurationException;
import org.mule.tooling.client.api.extension.ExtensionModelService;
import org.mule.tooling.client.api.tryit.TryItService;
import org.mule.tooling.client.internal.artifact.DefaultArtifactSerializationService;

import java.net.URL;
import java.util.Optional;
import java.util.function.Supplier;

/**
 * Implementation of a {@link ToolingRuntimeClient}. It uses SPI to look up for {@link RuntimeToolingService} implementation to
 * resolve tooling operations.
 *
 * @since 4.0
 */
class DefaultToolingRuntimeClient implements ToolingRuntimeClient {

  private final MavenClient mavenClient;
  private final ComponentBuildingDefinitionLoader componentBuildingDefinitionLoader;
  private final Optional<EmbeddedContainer> embeddedContainerOptional;
  private final Optional<AgentConfiguration> embeddedAgentConfigurationOptional;
  private AgentConfiguration agentConfiguration;
  private Supplier<RuntimeToolingService> runtimeToolingServiceSupplier;
  private RuntimeToolingService embeddedRuntimeToolingService;
  private RuntimeToolingService remoteRuntimeToolingService;

  private MuleRuntimeExtensionModelProvider muleRuntimeExtensionModelProvider;
  private ToolingExtensionModelAdapter toolingExtensionModelAdapter;
  private ArtifactSerializationService artifactSerializationService;
  private TryItService tryItService;

  /**
   * Creates an instance of the client without configuration.
   * 
   * @param runtimeToolingServiceSupplier {@link Supplier} for {@link RuntimeToolingService} to resolve operations through the
   *        Mule Runtime Agent. Not null.
   * @param mavenClient maven client
   * @param agentConfiguration agent configuration
   * @param muleRuntimeExtensionModelProvider provider to load {@link org.mule.runtime.api.meta.model.ExtensionModel}s
   * @param componentBuildingDefinitionLoader loader for {@link org.mule.runtime.dsl.api.component.ComponentBuildingDefinition}s
   * @param embeddedContainerOptional embedded container to use for runtime services
   */
  DefaultToolingRuntimeClient(Supplier<RuntimeToolingService> runtimeToolingServiceSupplier,
                              MavenClient mavenClient,
                              Optional<AgentConfiguration> agentConfiguration,
                              MuleRuntimeExtensionModelProvider muleRuntimeExtensionModelProvider,
                              ComponentBuildingDefinitionLoader componentBuildingDefinitionLoader,
                              Optional<EmbeddedContainer> embeddedContainerOptional,
                              Optional<AgentConfiguration> embeddedAgentConfigurationOptional) {
    checkNotNull(runtimeToolingServiceSupplier, "runtimeToolingServiceSupplier cannot be null");
    checkNotNull(mavenClient, "aetherDependencyResolver cannot be null");
    checkNotNull(agentConfiguration, "agentConfiguration cannot be null");
    checkNotNull(embeddedContainerOptional, "embeddedContainerOptional cannot be null");
    checkNotNull(embeddedAgentConfigurationOptional, "embeddedAgentConfigurationOptional cannot be null");
    checkState(!embeddedContainerOptional.isPresent() || embeddedAgentConfigurationOptional.isPresent(),
               "embeddedContainer configured but it's agent configuration is null");
    checkNotNull(muleRuntimeExtensionModelProvider, "muleRuntimeExtensionModelProvider cannot be null");

    this.mavenClient = mavenClient;
    this.runtimeToolingServiceSupplier = runtimeToolingServiceSupplier;
    this.agentConfiguration = agentConfiguration.orElse(null);
    this.componentBuildingDefinitionLoader = componentBuildingDefinitionLoader;
    this.embeddedContainerOptional = embeddedContainerOptional;
    this.embeddedAgentConfigurationOptional = embeddedAgentConfigurationOptional;

    // Services
    this.muleRuntimeExtensionModelProvider = muleRuntimeExtensionModelProvider;
    this.artifactSerializationService =
        new DefaultArtifactSerializationService(muleRuntimeExtensionModelProvider.getMuleExtensionModels(),
                                                muleRuntimeExtensionModelProvider,
                                                mavenClient);
    this.tryItService = new DefaultTryItService(lazyRemoteRuntimeToolingSupplier());

    initServices();

    // Tooling API Adapter
    this.toolingExtensionModelAdapter = new ToolingExtensionModelAdapter(muleRuntimeExtensionModelProvider);
  }

  private Supplier<RuntimeToolingService> lazyRuntimeToolingClientSupplier() {
    return () -> {
      if (embeddedRuntimeToolingService != null) {
        return embeddedRuntimeToolingService;
      }
      if (embeddedContainerOptional.isPresent()) {
        embeddedRuntimeToolingService = runtimeToolingServiceSupplier.get();
        embeddedRuntimeToolingService.setToolingApiUrl(embeddedAgentConfigurationOptional.get().getToolingApiURLSupplier().get(),
                                                       embeddedAgentConfigurationOptional.get().getDefaultConnectTimeout(),
                                                       embeddedAgentConfigurationOptional.get().getDefaultReadTimeout(), empty());
        return embeddedRuntimeToolingService;
      } else {
        return lazyRemoteRuntimeToolingSupplier().get();
      }
    };
  }

  private Supplier<RuntimeToolingService> lazyRemoteRuntimeToolingSupplier() {
    return () -> {
      if (remoteRuntimeToolingService != null) {
        return remoteRuntimeToolingService;
      }
      if (this.agentConfiguration == null) {
        throw new MissingToolingConfigurationException("Tooling Client has to be configure with a configuration to access REST Tooling API in order to resolve operations");
      }
      remoteRuntimeToolingService = this.runtimeToolingServiceSupplier.get();
      remoteRuntimeToolingService.setToolingApiUrl(this.agentConfiguration.getToolingApiURLSupplier().get(),
                                                   this.agentConfiguration.getDefaultConnectTimeout(),
                                                   this.agentConfiguration.getDefaultReadTimeout(),
                                                   this.agentConfiguration.getSSLContext());
      return remoteRuntimeToolingService;
    };
  }

  private void initServices() {
    initIfNeeded(muleRuntimeExtensionModelProvider);
    initIfNeeded(artifactSerializationService);
    initIfNeeded(tryItService);
  }

  private void initIfNeeded(Object service) {
    if (service instanceof Initializable) {
      ((Initializable) service).init();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ToolingArtifact newToolingArtifact(Supplier<URL> applicationUrlSupplier) {
    return new DefaultToolingArtifact(applicationUrlSupplier, lazyRuntimeToolingClientSupplier(), mavenClient,
                                      muleRuntimeExtensionModelProvider, componentBuildingDefinitionLoader);
  }

  @Override
  public ToolingArtifact fetchToolingArtifact(String applicationId, Supplier<URL> applicationUrlSupplier) {
    return new DefaultToolingArtifact(applicationId, applicationUrlSupplier, lazyRuntimeToolingClientSupplier(),
                                      mavenClient, muleRuntimeExtensionModelProvider, componentBuildingDefinitionLoader);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public TryItService tryItService() {
    return tryItService;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ConnectivityTestingServiceBuilder newConnectivityTestingService() {
    return new TemporaryConnectivityTestingServiceBuilder(lazyRuntimeToolingClientSupplier().get());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ExtensionModelService extensionModelService() {
    return toolingExtensionModelAdapter;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ArtifactSerializationService artifactSerializationService() {
    return artifactSerializationService;
  }

}
