/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static com.google.common.collect.Maps.newHashMap;
import static java.util.Collections.emptySet;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;
import org.mule.runtime.api.meta.model.util.ComponentModelVisitor;
import org.mule.runtime.extension.internal.property.MetadataKeyPartModelProperty;
import org.mule.tooling.client.api.extension.model.Category;
import org.mule.tooling.client.api.extension.model.DisplayModel;
import org.mule.tooling.client.api.extension.model.ErrorModel;
import org.mule.tooling.client.api.extension.model.ExpressionSupport;
import org.mule.tooling.client.api.extension.model.ExtensionModel;
import org.mule.tooling.client.api.extension.model.ExternalLibraryModel;
import org.mule.tooling.client.api.extension.model.ExternalLibraryType;
import org.mule.tooling.client.api.extension.model.ImportedTypeModel;
import org.mule.tooling.client.api.extension.model.LayoutModel;
import org.mule.tooling.client.api.extension.model.MuleVersion;
import org.mule.tooling.client.api.extension.model.OutputModel;
import org.mule.tooling.client.api.extension.model.ParameterDslConfiguration;
import org.mule.tooling.client.api.extension.model.PathModel;
import org.mule.tooling.client.api.extension.model.PathModelType;
import org.mule.tooling.client.api.extension.model.Stereotype;
import org.mule.tooling.client.api.extension.model.SubTypesModel;
import org.mule.tooling.client.api.extension.model.XmlDslModel;
import org.mule.tooling.client.api.extension.model.config.ConfigurationModel;
import org.mule.tooling.client.api.extension.model.connection.ConnectionManagementType;
import org.mule.tooling.client.api.extension.model.connection.ConnectionProviderModel;
import org.mule.tooling.client.api.extension.model.metadata.MetadataKeyPartModel;
import org.mule.tooling.client.api.extension.model.operation.OperationModel;
import org.mule.tooling.client.api.extension.model.operation.RouteModel;
import org.mule.tooling.client.api.extension.model.operation.RouterModel;
import org.mule.tooling.client.api.extension.model.operation.ScopeModel;
import org.mule.tooling.client.api.extension.model.parameter.ExclusiveParametersModel;
import org.mule.tooling.client.api.extension.model.parameter.ParameterGroupModel;
import org.mule.tooling.client.api.extension.model.parameter.ParameterModel;
import org.mule.tooling.client.api.extension.model.parameter.ParameterRole;
import org.mule.tooling.client.api.extension.model.source.SourceCallbackModel;
import org.mule.tooling.client.api.extension.model.source.SourceModel;

import com.google.common.collect.ImmutableSet;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

/**
 * Factory for {@link org.mule.tooling.client.api.extension.model.ExtensionModel}. Responsible for adapting a Mule Runtime
 * {@link org.mule.runtime.api.meta.model.ExtensionModel} and implements flyweight pattern in order to reduce the amount of objects created.
 *
 * @since 4.0
 */
public class ExtensionModelFactory {

  public ExtensionModel createExtensionModel(org.mule.runtime.api.meta.model.ExtensionModel runtimeExtensionModel) {
    FactoryContext factoryContext = new FactoryContext();
    return new ExtensionModel(
                              runtimeExtensionModel.getName(),
                              toCategoryDTO(runtimeExtensionModel.getCategory()),
                              new MuleVersion(runtimeExtensionModel.getMinMuleVersion().toCompleteNumericVersion()),
                              runtimeExtensionModel.getVendor(),
                              factoryContext.copy(runtimeExtensionModel.getResources()),
                              runtimeExtensionModel.getImportedTypes().stream()
                                  .map(importedTypeModel -> new ImportedTypeModel(importedTypeModel.getImportedType())).collect(
                                                                                                                                toSet()),
                              runtimeExtensionModel
                                  .getSubTypes().stream().map(subTypesModel -> new SubTypesModel(subTypesModel.getBaseType(),
                                                                                                 subTypesModel.getSubTypes()))
                                  .collect(
                                           toSet()),
                              runtimeExtensionModel.getTypes(),
                              runtimeExtensionModel.getVersion(),
                              new XmlDslModel(runtimeExtensionModel.getXmlDslModel().getXsdFileName(),
                                              runtimeExtensionModel.getXmlDslModel().getSchemaVersion(),
                                              runtimeExtensionModel.getXmlDslModel().getPrefix(),
                                              runtimeExtensionModel.getXmlDslModel().getNamespace(),
                                              runtimeExtensionModel.getXmlDslModel().getSchemaLocation()),
                              runtimeExtensionModel.getDescription(),
                              factoryContext.toDisplayModelTO(runtimeExtensionModel.getDisplayModel()),
                              factoryContext.toExternalLibraryModelsDTO(runtimeExtensionModel.getExternalLibraryModels()),
                              factoryContext.toConfigurationModelsDTO(runtimeExtensionModel),
                              factoryContext.toOperationModelsDTO(runtimeExtensionModel.getOperationModels()),
                              factoryContext.toConnectionProvidersDTO(runtimeExtensionModel.getConnectionProviders()),
                              factoryContext.toSourceModelsDTO(runtimeExtensionModel.getSourceModels()));
  }

  private Category toCategoryDTO(org.mule.runtime.api.meta.Category category) {
    if (category == org.mule.runtime.api.meta.Category.COMMUNITY) {
      return new Category(category.name(), false, true, false, false, false);
    }
    if (category == org.mule.runtime.api.meta.Category.SELECT) {
      return new Category(category.name(), false, false, true, false, false);
    }
    if (category == org.mule.runtime.api.meta.Category.PREMIUM) {
      return new Category(category.name(), false, false, false, true, false);
    }
    if (category == org.mule.runtime.api.meta.Category.CERTIFIED) {
      return new Category(category.name(), false, false, false, false, true);
    }
    return new Category(category.name(), true, false, false, false, false);
  }

  private class FactoryContext {

    // Caches for implementing flyweight pattern
    private Map<String, ConnectionProviderModel> connectionProviderModelsCache = newHashMap();
    private Map<String, SourceModel> sourceModelsCache = newHashMap();
    private Map<String, OperationModel> operationModelsCache = newHashMap();
    private Map<String, ScopeModel> scopeModelsCache = newHashMap();
    private Map<String, RouterModel> routerModelsCache = newHashMap();

    private <T> Set<T> copy(Set<T> values) {
      return values != null ? ImmutableSet.copyOf(values) : ImmutableSet.of();
    }

    private Set<ExternalLibraryModel> toExternalLibraryModelsDTO(
                                                                 Set<org.mule.runtime.api.meta.model.ExternalLibraryModel> externalLibraryModels) {
      return externalLibraryModels.stream().map(externalLibraryModel -> new ExternalLibraryModel(externalLibraryModel.getName(),
                                                                                                 externalLibraryModel
                                                                                                     .getDescription(),
                                                                                                 externalLibraryModel
                                                                                                     .getRequiredClassName()
                                                                                                     .orElse(null),
                                                                                                 externalLibraryModel
                                                                                                     .getRegexMatcher()
                                                                                                     .orElse(null),
                                                                                                 toExternalLibraryTypeDTO(externalLibraryModel
                                                                                                     .getType())))
          .collect(toSet());
    }

    private ExternalLibraryType toExternalLibraryTypeDTO(org.mule.runtime.api.meta.ExternalLibraryType type) {
      if (type == org.mule.runtime.api.meta.ExternalLibraryType.NATIVE) {
        return new ExternalLibraryType(type.name(), false, true, false, false);
      }
      if (type == org.mule.runtime.api.meta.ExternalLibraryType.JAR) {
        return new ExternalLibraryType(type.name(), false, false, true, false);
      }
      if (type == org.mule.runtime.api.meta.ExternalLibraryType.DEPENDENCY) {
        return new ExternalLibraryType(type.name(), false, false, false, true);
      }
      return new ExternalLibraryType(type.name(), true, false, false, false);
    }

    public List<ConfigurationModel> toConfigurationModelsDTO(
                                                             org.mule.runtime.api.meta.model.ExtensionModel serializedExtensionModel) {
      return serializedExtensionModel.getConfigurationModels().stream()
          .map(configurationModel -> new ConfigurationModel(configurationModel.getName(),
                                                            configurationModel.getDescription(),
                                                            toParameterGroupModelsDTO(configurationModel
                                                                .getParameterGroupModels()),
                                                            toOperationModelsDTO(configurationModel.getOperationModels()),
                                                            toExternalLibraryModelsDTO(configurationModel
                                                                .getExternalLibraryModels()),
                                                            toSourceModelsDTO(configurationModel.getSourceModels()),
                                                            toConnectionProvidersDTO(configurationModel.getConnectionProviders()),
                                                            toDisplayModelTO(configurationModel.getDisplayModel())))
          .collect(toList());
    }

    public List<ConnectionProviderModel> toConnectionProvidersDTO(List<org.mule.runtime.api.meta.model.connection.ConnectionProviderModel> connectionProviders) {
      return connectionProviders.stream()
          .map(connectionProviderModel -> {
            if (!connectionProviderModelsCache.containsKey(connectionProviderModel.getName())) {
              connectionProviderModelsCache.put(connectionProviderModel.getName(),
                                                new ConnectionProviderModel(connectionProviderModel.getName(),
                                                                            connectionProviderModel.getDescription(),
                                                                            toDisplayModelTO(connectionProviderModel
                                                                                .getDisplayModel()),
                                                                            toConnectionManagementTypeDTO(connectionProviderModel
                                                                                .getConnectionManagementType()),
                                                                            toParameterGroupModelsDTO(connectionProviderModel
                                                                                .getParameterGroupModels()),
                                                                            toExternalLibraryModelsDTO(connectionProviderModel
                                                                                .getExternalLibraryModels())));
            }

            return connectionProviderModelsCache.get(connectionProviderModel.getName());
          })
          .collect(toList());
    }

    private ConnectionManagementType toConnectionManagementTypeDTO(
                                                                   org.mule.runtime.api.meta.model.connection.ConnectionManagementType connectionManagementType) {
      if (connectionManagementType == org.mule.runtime.api.meta.model.connection.ConnectionManagementType.POOLING) {
        return new ConnectionManagementType(connectionManagementType.name(), false, true, false, false);
      }
      if (connectionManagementType == org.mule.runtime.api.meta.model.connection.ConnectionManagementType.CACHED) {
        return new ConnectionManagementType(connectionManagementType.name(), false, false, true, false);
      }
      if (connectionManagementType == org.mule.runtime.api.meta.model.connection.ConnectionManagementType.NONE) {
        return new ConnectionManagementType(connectionManagementType.name(), false, false, false, true);
      }

      return new ConnectionManagementType(connectionManagementType.name(), true, false, false, false);
    }

    public List<SourceModel> toSourceModelsDTO(List<org.mule.runtime.api.meta.model.source.SourceModel> sourceModels) {
      return sourceModels.stream()
          .map(sourceModel -> toSourceModelDTO(sourceModel))
          .collect(toList());
    }

    private SourceModel toSourceModelDTO(org.mule.runtime.api.meta.model.source.SourceModel sourceModel) {
      if (!sourceModelsCache.containsKey(sourceModel.getName())) {
        sourceModelsCache.put(sourceModel.getName(), new SourceModel(sourceModel.getName(),
                                                                     sourceModel.getDescription(),
                                                                     toParameterGroupModelsDTO(sourceModel
                                                                         .getParameterGroupModels()),
                                                                     toDisplayModelTO(
                                                                                      sourceModel
                                                                                          .getDisplayModel()),
                                                                     toOutputModelDTO(sourceModel.getOutput()),
                                                                     toOutputModelDTO(
                                                                                      sourceModel
                                                                                          .getOutputAttributes()),
                                                                     sourceModel.isTransactional(),
                                                                     sourceModel.requiresConnection(),
                                                                     sourceModel.supportsStreaming(),
                                                                     toStereotypesDTO(
                                                                                      sourceModel
                                                                                          .getStereotypes()),
                                                                     sourceModel.hasResponse(),
                                                                     toSourceCallbackModelDTO(
                                                                                              sourceModel
                                                                                                  .getSuccessCallback()),
                                                                     toSourceCallbackModelDTO(
                                                                                              sourceModel
                                                                                                  .getErrorCallback()),
                                                                     toSourceCallbackModelDTO(
                                                                                              sourceModel
                                                                                                  .getTerminateCallback()),
                                                                     toErrorModelsDTO(
                                                                                      sourceModel
                                                                                          .getErrorModels())));
      }
      return sourceModelsCache.get(sourceModel.getName());
    }

    private Set<ErrorModel> toErrorModelsDTO(Set<org.mule.runtime.api.meta.model.error.ErrorModel> errorModels) {
      return errorModels.stream()
          .map(errorModel -> toErrorModelDTO(errorModel))
          .collect(toSet());
    }

    private ErrorModel toErrorModelDTO(org.mule.runtime.api.meta.model.error.ErrorModel errorModel) {
      if (errorModel == null) {
        return null;
      }
      return new ErrorModel(errorModel.getType(), errorModel.getNamespace(),
                            toErrorModelDTO(errorModel.getParent().orElse(null)));
    }

    private SourceCallbackModel toSourceCallbackModelDTO(Optional<org.mule.runtime.api.meta.model.source.SourceCallbackModel> sourceCallbackModelOptional) {
      return sourceCallbackModelOptional
          .map(sourceCallbackModel -> new SourceCallbackModel(sourceCallbackModel.getName(),
                                                              sourceCallbackModel.getDescription(),
                                                              toParameterGroupModelsDTO(sourceCallbackModel
                                                                  .getParameterGroupModels()),
                                                              toDisplayModelTO(sourceCallbackModel.getDisplayModel())))
          .orElse(null);
    }

    private Set<Stereotype> toStereotypesDTO(Set<org.mule.runtime.api.meta.model.Stereotype> stereotypes) {
      // Mule Extension Models (json written by hand) has null value for this property
      if (stereotypes == null) {
        return emptySet();
      }
      return stereotypes.stream()
          .map(stereotype -> new Stereotype(stereotype.getName()))
          .collect(toSet());
    }

    private OutputModel toOutputModelDTO(org.mule.runtime.api.meta.model.OutputModel output) {
      return new OutputModel(output.getDescription(), output.getType(), output.hasDynamicType());
    }

    public List<OperationModel> toOperationModelsDTO(List<org.mule.runtime.api.meta.model.operation.OperationModel> operationModels) {
      return operationModels.stream()
          .map(operationModel -> {
            ComponentModelVisitorDTO visitor = new ComponentModelVisitorDTO();
            operationModel.accept(visitor);
            return visitor.operationModelDTO;
          })
          .collect(toList());
    }

    private List<RouteModel> toRouteModelsDTO(List<org.mule.runtime.api.meta.model.operation.RouteModel> routeModels) {
      return routeModels.stream()
          .map(routeModel -> new RouteModel(routeModel.getName(),
                                            routeModel.getDescription(),
                                            routeModel.getMinOccurs(),
                                            routeModel.getMaxOccurs().orElse(null),
                                            toStereotypesDTO(routeModel.getAllowedStereotypes().orElse(emptySet())),
                                            toParameterGroupModelsDTO(routeModel.getParameterGroupModels())))
          .collect(toList());
    }

    private List<ParameterGroupModel> toParameterGroupModelsDTO(List<org.mule.runtime.api.meta.model.parameter.ParameterGroupModel> parameterGroupModels) {
      return parameterGroupModels.stream()
          .map(parameterGroupModel -> new ParameterGroupModel(parameterGroupModel.getName(),
                                                              parameterGroupModel.getDescription(),
                                                              toDisplayModelTO(parameterGroupModel.getDisplayModel()),
                                                              toLayoutModelDTO(parameterGroupModel.getLayoutModel()),
                                                              toParameterModelsDTO(parameterGroupModel.getParameterModels()),
                                                              toExclusiveParameterModelsDTO(parameterGroupModel
                                                                  .getExclusiveParametersModels()),
                                                              parameterGroupModel.isShowInDsl()))
          .collect(toList());
    }

    private List<ExclusiveParametersModel> toExclusiveParameterModelsDTO(List<org.mule.runtime.api.meta.model.parameter.ExclusiveParametersModel> exclusiveParametersModels) {
      return exclusiveParametersModels.stream()
          .map(exclusiveParametersModel -> new ExclusiveParametersModel(copy(exclusiveParametersModel
              .getExclusiveParameterNames()),
                                                                        exclusiveParametersModel.isOneRequired()))
          .collect(toList());
    }

    private List<ParameterModel> toParameterModelsDTO(List<org.mule.runtime.api.meta.model.parameter.ParameterModel> parameterModels) {
      return parameterModels.stream()
          .map(parameterModel -> new ParameterModel(parameterModel.getName(),
                                                    parameterModel.getDescription(),
                                                    toDisplayModelTO(parameterModel.getDisplayModel()),
                                                    parameterModel.getType(),
                                                    parameterModel.hasDynamicType(),
                                                    parameterModel.isRequired(),
                                                    parameterModel.isOverrideFromConfig(),
                                                    toExpressionSupportDTO(parameterModel.getExpressionSupport()),
                                                    parameterModel.getDefaultValue(),
                                                    toParameterRoleDTO(parameterModel.getRole()),
                                                    toParameterDslConfigurationDTO(parameterModel.getDslConfiguration()),
                                                    toLayoutModelDTO(parameterModel.getLayoutModel()),
                                                    toMetadataKeyPartModel(parameterModel.getModelProperty(
                                                                                                           MetadataKeyPartModelProperty.class))))
          .collect(toList());
    }

    private MetadataKeyPartModel toMetadataKeyPartModel(Optional<MetadataKeyPartModelProperty> modelProperty) {
      return modelProperty.map(metadataKeyPartModelProperty -> new MetadataKeyPartModel(metadataKeyPartModelProperty.getOrder()))
          .orElse(null);
    }

    private ExpressionSupport toExpressionSupportDTO(org.mule.runtime.api.meta.ExpressionSupport expressionSupport) {
      if (expressionSupport == org.mule.runtime.api.meta.ExpressionSupport.SUPPORTED) {
        return new ExpressionSupport(expressionSupport.name(), false, true, false, false);
      }
      if (expressionSupport == org.mule.runtime.api.meta.ExpressionSupport.NOT_SUPPORTED) {
        return new ExpressionSupport(expressionSupport.name(), false, false, true, false);
      }
      if (expressionSupport == org.mule.runtime.api.meta.ExpressionSupport.REQUIRED) {
        return new ExpressionSupport(expressionSupport.name(), false, false, false, true);
      }
      return new ExpressionSupport(expressionSupport.name(), true, false, false, false);
    }

    private ParameterDslConfiguration toParameterDslConfigurationDTO(
                                                                     org.mule.runtime.api.meta.model.ParameterDslConfiguration dslConfiguration) {
      return new ParameterDslConfiguration(dslConfiguration.allowsInlineDefinition(), dslConfiguration.allowTopLevelDefinition(),
                                           dslConfiguration.allowsReferences());
    }

    private LayoutModel toLayoutModelDTO(Optional<org.mule.runtime.api.meta.model.display.LayoutModel> layoutModelOptional) {
      return layoutModelOptional.map(layoutModel -> new LayoutModel(layoutModel.getTabName().orElse(null),
                                                                    layoutModel.getOrder().orElse(null),
                                                                    layoutModel.isPassword(),
                                                                    layoutModel.isText(),
                                                                    layoutModel.isQuery()))
          .orElse(null);
    }

    private ParameterRole toParameterRoleDTO(org.mule.runtime.api.meta.model.parameter.ParameterRole parameterRole) {
      if (parameterRole == org.mule.runtime.api.meta.model.parameter.ParameterRole.BEHAVIOUR) {
        return new ParameterRole(parameterRole.name(), false, true, false, false);
      }
      if (parameterRole == org.mule.runtime.api.meta.model.parameter.ParameterRole.CONTENT) {
        return new ParameterRole(parameterRole.name(), false, false, true, false);
      }
      if (parameterRole == org.mule.runtime.api.meta.model.parameter.ParameterRole.PRIMARY_CONTENT) {
        return new ParameterRole(parameterRole.name(), false, false, false, true);
      }
      return new ParameterRole(parameterRole.name(), true, false, false, false);
    }

    public DisplayModel toDisplayModelTO(Optional<org.mule.runtime.api.meta.model.display.DisplayModel> displayModelOptional) {
      return displayModelOptional.map(displayModel -> new DisplayModel(displayModel.getDisplayName(), displayModel.getSummary(),
                                                                       displayModel.getExample(),
                                                                       toPathModelDTO(displayModel.getPathModel())))
          .orElse(null);
    }

    private PathModel toPathModelDTO(Optional<org.mule.runtime.api.meta.model.display.PathModel> pathModelOptional) {
      return pathModelOptional.map(pathModel -> new PathModel(toPathModelTypeDTO(pathModel.getType()), pathModel.acceptsUrls(),
                                                              pathModel.getFileExtensions().toArray(new String[0])))
          .orElse(null);
    }

    private PathModelType toPathModelTypeDTO(org.mule.runtime.api.meta.model.display.PathModel.Type type) {
      if (type == org.mule.runtime.api.meta.model.display.PathModel.Type.DIRECTORY) {
        return new PathModelType(type.name(), false, true, false, false);
      }
      if (type == org.mule.runtime.api.meta.model.display.PathModel.Type.FILE) {
        return new PathModelType(type.name(), false, false, true, false);
      }
      if (type == org.mule.runtime.api.meta.model.display.PathModel.Type.ANY) {
        return new PathModelType(type.name(), false, false, false, true);
      }
      return new PathModelType(type.name(), true, false, false, false);
    }

    private class ComponentModelVisitorDTO implements ComponentModelVisitor {

      protected OperationModel operationModelDTO;

      @Override
      public void visit(org.mule.runtime.api.meta.model.operation.OperationModel operationModel) {
        if (!operationModelsCache.containsKey(operationModel.getName())) {
          operationModelsCache.put(operationModel.getName(), new OperationModel(operationModel.getName(),
                                                                                operationModel.getDescription(),
                                                                                toParameterGroupModelsDTO(operationModel
                                                                                    .getParameterGroupModels()),
                                                                                toDisplayModelTO(operationModel
                                                                                    .getDisplayModel()),
                                                                                operationModel.isBlocking(),
                                                                                toErrorModelsDTO(operationModel
                                                                                    .getErrorModels()),
                                                                                toOutputModelDTO(operationModel
                                                                                    .getOutput()),
                                                                                toOutputModelDTO(operationModel
                                                                                    .getOutputAttributes()),
                                                                                operationModel.isTransactional(),
                                                                                operationModel.requiresConnection(),
                                                                                operationModel.supportsStreaming(),
                                                                                toStereotypesDTO(operationModel
                                                                                    .getStereotypes())));
        }
        operationModelDTO = operationModelsCache.get(operationModel.getName());
      }

      @Override
      public void visit(org.mule.runtime.api.meta.model.operation.ScopeModel scopeModel) {
        if (!scopeModelsCache.containsKey(scopeModel.getName())) {
          scopeModelsCache.put(scopeModel.getName(), new ScopeModel(scopeModel.getName(),
                                                                    scopeModel.getDescription(),
                                                                    toParameterGroupModelsDTO(scopeModel
                                                                        .getParameterGroupModels()),
                                                                    toDisplayModelTO(scopeModel
                                                                        .getDisplayModel()),
                                                                    scopeModel.isBlocking(),
                                                                    toErrorModelsDTO(scopeModel
                                                                        .getErrorModels()),
                                                                    toOutputModelDTO(scopeModel
                                                                        .getOutput()),
                                                                    toOutputModelDTO(scopeModel
                                                                        .getOutputAttributes()),
                                                                    scopeModel.isTransactional(),
                                                                    scopeModel.requiresConnection(),
                                                                    scopeModel.supportsStreaming(),
                                                                    toStereotypesDTO(scopeModel
                                                                        .getStereotypes())));
        }
        operationModelDTO = scopeModelsCache.get(scopeModel.getName());
      }

      @Override
      public void visit(org.mule.runtime.api.meta.model.operation.RouterModel routerModel) {
        if (!routerModelsCache.containsKey(routerModel.getName())) {
          routerModelsCache.put(routerModel.getName(), new RouterModel(routerModel.getName(),
                                                                       routerModel.getDescription(),
                                                                       toParameterGroupModelsDTO(routerModel
                                                                           .getParameterGroupModels()),
                                                                       toDisplayModelTO(routerModel
                                                                           .getDisplayModel()),
                                                                       routerModel.isBlocking(),
                                                                       toErrorModelsDTO(routerModel
                                                                           .getErrorModels()),
                                                                       toOutputModelDTO(routerModel
                                                                           .getOutput()),
                                                                       toOutputModelDTO(routerModel
                                                                           .getOutputAttributes()),
                                                                       routerModel.isTransactional(),
                                                                       routerModel.requiresConnection(),
                                                                       routerModel.supportsStreaming(),
                                                                       toStereotypesDTO(routerModel
                                                                           .getStereotypes()),
                                                                       toRouteModelsDTO(routerModel
                                                                           .getRouteModels())));
        }
        operationModelDTO = routerModelsCache.get(routerModel.getName());
      }

      @Override
      public void visit(org.mule.runtime.api.meta.model.source.SourceModel sourceModel) {
        // Nothing to do here...
        operationModelDTO = null;
      }
    }
  }

}
