/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.stream.Collectors.toSet;

import org.mule.runtime.api.metadata.MetadataKey;
import org.mule.runtime.api.metadata.MetadataKeysContainer;
import org.mule.runtime.api.metadata.resolving.MetadataComponent;
import org.mule.runtime.api.metadata.resolving.MetadataFailure;
import org.mule.runtime.api.metadata.resolving.MetadataResult;
import org.mule.tooling.client.api.exception.ServiceUnavailableException;
import org.mule.tooling.client.api.exception.ToolingException;
import org.mule.tooling.client.api.metadata.MetadataKeysRequest;
import org.mule.tooling.client.api.metadata.MetadataService;

import org.mule.tooling.client.internal.application.Deployable;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Public implementation of {@link MetadataService} that adapts Mule-API objects to DTOs
 *
 * @since 4.0
 */
public class ToolingMetadataServiceAdapter implements MetadataService {

  private final Logger logger = LoggerFactory.getLogger(this.getClass());

  private MetadataProvider metadataProvider;

  /**
   * Creates an instance of this service adapter.
   *
   * @param metadataProvider {@link Supplier} for a {@link Deployable}. Non null. Should return always the same instance.
   */
  public ToolingMetadataServiceAdapter(MetadataProvider metadataProvider) {
    checkNotNull(metadataProvider, "metadataProvider cannot be null");

    this.metadataProvider = metadataProvider;
  }

  private static org.mule.tooling.client.api.metadata.dto.MetadataKey metadataKeyToDTO(MetadataKey metadataKey) {
    if (metadataKey == null) {
      return null;
    }
    Set<org.mule.tooling.client.api.metadata.dto.MetadataKey> children =
        metadataKey.getChilds().stream().map(ToolingMetadataServiceAdapter::metadataKeyToDTO).collect(toSet());

    return new org.mule.tooling.client.api.metadata.dto.MetadataKey(metadataKey.getId(), metadataKey.getDisplayName(),
                                                                    children, metadataKey.getPartName());
  }

  private static org.mule.tooling.client.api.metadata.dto.MetadataKeysContainer toMetadataKeysContainerDTO(MetadataKeysContainer container) {
    Map<String, Set<org.mule.runtime.api.metadata.MetadataKey>> map = container.getKeysByCategory();

    Map<String, Set<org.mule.tooling.client.api.metadata.dto.MetadataKey>> dtomap = new HashMap<>();

    for (String key : map.keySet()) {
      Set<org.mule.tooling.client.api.metadata.dto.MetadataKey> set =
          map.get(key).stream().map(ToolingMetadataServiceAdapter::metadataKeyToDTO).collect(toSet());
      dtomap.put(key, set);
    }
    return new org.mule.tooling.client.api.metadata.dto.MetadataKeysContainer(dtomap);
  }

  private static org.mule.tooling.client.api.metadata.dto.MetadataComponent toMetadataComponentDTO(MetadataComponent component) {
    if (component.equals(MetadataComponent.OUTPUT_ATTRIBUTES)) {
      return new org.mule.tooling.client.api.metadata.dto.MetadataComponent(component.name(), false, true, false, false, false,
                                                                            false, false);
    } else if (component.equals(MetadataComponent.OUTPUT_PAYLOAD)) {
      return new org.mule.tooling.client.api.metadata.dto.MetadataComponent(component.name(), false, false, true, false, false,
                                                                            false, false);
    } else if (component.equals(MetadataComponent.INPUT)) {
      return new org.mule.tooling.client.api.metadata.dto.MetadataComponent(component.name(), false, false, false, true, false,
                                                                            false, false);
    } else if (component.equals(MetadataComponent.COMPONENT)) {
      return new org.mule.tooling.client.api.metadata.dto.MetadataComponent(component.name(), false, false, false, false, true,
                                                                            false, false);
    } else if (component.equals(MetadataComponent.ENTITY)) {
      return new org.mule.tooling.client.api.metadata.dto.MetadataComponent(component.name(), false, false, false, false, false,
                                                                            true, false);
    } else if (component.equals(MetadataComponent.KEYS)) {
      return new org.mule.tooling.client.api.metadata.dto.MetadataComponent(component.name(), false, false, false, false, false,
                                                                            false, true);
    } else {
      return new org.mule.tooling.client.api.metadata.dto.MetadataComponent(component.name(), true, false, false, false, false,
                                                                            false, false);
    }
  }

  private static org.mule.tooling.client.api.metadata.dto.MetadataResult toMetadataResultDTO(MetadataResult result,
                                                                                             Supplier<Object> supplier) {
    if (result.isSuccess()) {
      return org.mule.tooling.client.api.metadata.dto.MetadataResult.success(supplier.get());
    } else {
      List<MetadataFailure> failures = result.getFailures();

      return org.mule.tooling.client.api.metadata.dto.MetadataResult
          .failure(failures.stream()
              .map(failure -> new org.mule.tooling.client.api.metadata.dto.MetadataFailure(toMetadataComponentDTO(failure
                  .getFailingComponent()),
                                                                                           failure
                                                                                               .getFailingElement()
                                                                                               .orElse(""),
                                                                                           failure
                                                                                               .getMessage(),
                                                                                           failure
                                                                                               .getReason(),
                                                                                           new org.mule.tooling.client.api.metadata.dto.FailureCode(failure
                                                                                               .getFailureCode()
                                                                                               .getName())))
              .collect(Collectors.toList()));
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public org.mule.tooling.client.api.metadata.dto.MetadataResult<org.mule.tooling.client.api.metadata.dto.MetadataKeysContainer> getMetadataKeys(MetadataKeysRequest metadataKeysRequest)
      throws ServiceUnavailableException, ToolingException {

    MetadataResult<MetadataKeysContainer> metadataKeys = metadataProvider.getMetadataKeys(metadataKeysRequest);
    return toMetadataResultDTO(metadataKeys, () -> toMetadataKeysContainerDTO(metadataKeys.get()));
  }

}
