/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal.application;

import static com.google.common.base.Preconditions.checkArgument;
import static java.util.Objects.requireNonNull;
import static org.apache.commons.io.FileUtils.toFile;
import org.mule.tooling.agent.RuntimeToolingService;
import org.mule.tooling.client.api.exception.ToolingException;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.net.URL;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Handles a reference to the exploded application from file system.
 *
 * @since 4.0
 */
public class FileSystemApplicationDeployer implements ApplicationDeployer {

  private final Logger logger = LoggerFactory.getLogger(this.getClass());
  private URL applicationUrl;
  private RuntimeToolingService runtimeToolingService;

  public FileSystemApplicationDeployer(URL applicationUrl, RuntimeToolingService runtimeToolingService) {
    requireNonNull(applicationUrl, "applicationUrl cannot be null");
    requireNonNull(runtimeToolingService, "runtimeToolingService cannot be null");

    this.applicationUrl = applicationUrl;
    this.runtimeToolingService = runtimeToolingService;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String deploy() {
    if (logger.isDebugEnabled()) {
      logger.debug("Deploying application from URL: {}", applicationUrl);
    }

    File appLocation = toFile(applicationUrl);
    checkArgument(appLocation != null && appLocation.exists(), "applicationUrl: " + applicationUrl + " is invalid");

    String applicationId;
    if (appLocation.isDirectory()) {
      applicationId = runtimeToolingService.deployApplication(appLocation);
    } else if (appLocation.isFile()) {
      try {
        applicationId = runtimeToolingService.deployApplication(new FileInputStream(appLocation));
      } catch (IOException e) {
        throw new ToolingException("Couldn't create inputStream for zip file from application URL: " + applicationUrl);
      }
    } else {
      throw new IllegalArgumentException("Couldn't handle URL for deploying application: " + applicationUrl);
    }
    return applicationId;
  }

}
